package dev.emi.trinkets.data;

import java.io.IOException;
import java.io.InputStreamReader;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonSyntaxException;

import dev.emi.trinkets.TrinketsMain;
import dev.emi.trinkets.api.SlotType;
import dev.emi.trinkets.api.TrinketEnums.DropRule;
import dev.emi.trinkets.data.SlotLoader.GroupData;
import net.fabricmc.fabric.api.resource.IdentifiableResourceReloadListener;
import net.minecraft.class_2960;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import net.minecraft.class_3518;
import net.minecraft.class_3695;
import net.minecraft.class_4080;

public class SlotLoader extends class_4080<Map<String, GroupData>> implements IdentifiableResourceReloadListener {

	public static final SlotLoader INSTANCE = new SlotLoader();

	static final class_2960 ID = class_2960.method_60655(TrinketsMain.MOD_ID, "slots");

	private static final Gson GSON = (new GsonBuilder()).setPrettyPrinting().disableHtmlEscaping().create();
	private static final int FILE_SUFFIX_LENGTH = ".json".length();

	private Map<String, GroupData> slots = new HashMap<>();

	@Override
	protected Map<String, GroupData> method_18789(class_3300 resourceManager, class_3695 profiler) {
		Map<String, GroupData> map = new HashMap<>();
		String dataType = "slots";
		for (Map.Entry<class_2960, List<class_3298>> entry : resourceManager.method_41265(dataType, id -> id.method_12832().endsWith(".json")).entrySet()) {
			class_2960 identifier = entry.getKey();

			if (identifier.method_12836().equals(TrinketsMain.MOD_ID)) {

				try {
					for (class_3298 resource : entry.getValue()) {
						InputStreamReader reader = new InputStreamReader(resource.method_14482());
						JsonObject jsonObject = class_3518.method_15276(GSON, reader, JsonObject.class);

						if (jsonObject != null) {
							String path = identifier.method_12832();
							String[] parsed = path.substring(dataType.length() + 1, path.length() - FILE_SUFFIX_LENGTH).split("/");
							String groupName = parsed[0];
							String fileName = parsed[parsed.length - 1];
							GroupData group = map.computeIfAbsent(groupName, (k) -> new GroupData());

							try {
								if (fileName.equals("group")) {
									group.read(jsonObject);
								} else {
									SlotData slot = group.slots.computeIfAbsent(fileName, (k) -> new SlotData());
									slot.read(jsonObject);
								}
							} catch (JsonSyntaxException e) {
								TrinketsMain.LOGGER.error("[trinkets] Syntax error while reading data for " + path);
								e.printStackTrace();
							}
						}
					}
				} catch (IOException e) {
					TrinketsMain.LOGGER.error("[trinkets] Unknown IO error while reading slot data!");
					e.printStackTrace();
				}
			}
		}
		return map;
	}

	@Override
	protected void apply(Map<String, GroupData> loader, class_3300 manager, class_3695 profiler) {
		this.slots = loader;
	}

	public Map<String, GroupData> getSlots() {
		return ImmutableMap.copyOf(this.slots);
	}

	@Override
	public class_2960 getFabricId() {
		return ID;
	}

	static class GroupData {

		private int slotId = -1;
		private int order = 0;
		private final Map<String, SlotData> slots = new HashMap<>();

		void read(JsonObject jsonObject) {
			slotId = class_3518.method_15282(jsonObject, "slot_id", slotId);
			order = class_3518.method_15282(jsonObject, "order", order);
		}

		int getSlotId() {
			return slotId;
		}

		int getOrder() {
			return order;
		}

		SlotData getSlot(String name) {
			return slots.get(name);
		}
	}

	static class SlotData {
		private static final Set<class_2960> DEFAULT_QUICK_MOVE_PREDICATES = ImmutableSet.of(class_2960.method_60655("trinkets", "all"));
		private static final Set<class_2960> DEFAULT_VALIDATOR_PREDICATES = ImmutableSet.of(class_2960.method_60655("trinkets", "tag"));
		private static final Set<class_2960> DEFAULT_TOOLTIP_PREDICATES = ImmutableSet.of(class_2960.method_60655("trinkets", "all"));

		private int order = 0;
		private int amount = -1;
		private String icon = "";
		private final Set<String> quickMovePredicates = new HashSet<>();
		private final Set<String> validatorPredicates = new HashSet<>();
		private final Set<String> tooltipPredicates = new HashSet<>();
		private String dropRule = DropRule.DEFAULT.toString();

		SlotType create(String group, String name) {
			class_2960 finalIcon = class_2960.method_60654(icon);
			finalIcon = class_2960.method_60655(finalIcon.method_12836(), "textures/" + finalIcon.method_12832() + ".png");
			Set<class_2960> finalValidatorPredicates = validatorPredicates.stream().map(class_2960::method_60654).collect(Collectors.toSet());
			Set<class_2960> finalQuickMovePredicates = quickMovePredicates.stream().map(class_2960::method_60654).collect(Collectors.toSet());
			Set<class_2960> finalTooltipPredicates = tooltipPredicates.stream().map(class_2960::method_60654).collect(Collectors.toSet());
			if (finalValidatorPredicates.isEmpty()) {
				finalValidatorPredicates = DEFAULT_VALIDATOR_PREDICATES;
			}
			if (finalQuickMovePredicates.isEmpty()) {
				finalQuickMovePredicates = DEFAULT_QUICK_MOVE_PREDICATES;
			}
			if (finalTooltipPredicates.isEmpty()) {
				finalTooltipPredicates = DEFAULT_TOOLTIP_PREDICATES;
			}
			if (amount == -1) {
				amount = 1;
			}
			return new SlotType(group, name, order, amount, finalIcon, finalQuickMovePredicates, finalValidatorPredicates,
				finalTooltipPredicates, DropRule.valueOf(dropRule));
		}

		void read(JsonObject jsonObject) {
			boolean replace = class_3518.method_15258(jsonObject, "replace", false);

			order = class_3518.method_15282(jsonObject, "order", order);

			int jsonAmount = class_3518.method_15282(jsonObject, "amount", amount);
			amount = replace ? jsonAmount : Math.max(jsonAmount, amount);

			icon = class_3518.method_15253(jsonObject, "icon", icon);

			JsonArray jsonQuickMovePredicates = class_3518.method_15292(jsonObject, "quick_move_predicates", new JsonArray());

			if (jsonQuickMovePredicates != null) {

				if (replace && jsonQuickMovePredicates.size() > 0) {
					quickMovePredicates.clear();
				}

				for (JsonElement jsonQuickMovePredicate : jsonQuickMovePredicates) {
					quickMovePredicates.add(jsonQuickMovePredicate.getAsString());
				}
			}

			String jsonDropRule = class_3518.method_15253(jsonObject, "drop_rule", dropRule).toUpperCase();

			if (DropRule.has(jsonDropRule)) {
				dropRule = jsonDropRule;
			}
			JsonArray jsonValidatorPredicates = class_3518.method_15292(jsonObject, "validator_predicates", new JsonArray());

			if (jsonValidatorPredicates != null) {

				if (replace && jsonValidatorPredicates.size() > 0) {
					validatorPredicates.clear();
				}

				for (JsonElement jsonValidatorPredicate : jsonValidatorPredicates) {
					validatorPredicates.add(jsonValidatorPredicate.getAsString());
				}
			}

			JsonArray jsonTooltipPredicates = class_3518.method_15292(jsonObject, "tooltip_predicates", new JsonArray());

			if (jsonTooltipPredicates != null) {

				if (replace && jsonTooltipPredicates.size() > 0) {
					tooltipPredicates.clear();
				}

				for (JsonElement jsonTooltipPredicate : jsonTooltipPredicates) {
					tooltipPredicates.add(jsonTooltipPredicate.getAsString());
				}
			}
		}
	}
}
