package dev.emi.trinkets.api;

import java.util.ArrayList;

import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;

import dev.emi.trinkets.mixin.accessor.LivingEntityAccessor;
import java.util.function.Consumer;
import net.minecraft.class_10192;
import net.minecraft.class_1309;
import net.minecraft.class_1320;
import net.minecraft.class_1322;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_1890;
import net.minecraft.class_2960;
import net.minecraft.class_3218;
import net.minecraft.class_3222;
import net.minecraft.class_3414;
import net.minecraft.class_3417;
import net.minecraft.class_6880;
import net.minecraft.class_7923;
import net.minecraft.class_9334;
import net.minecraft.class_9701;

public interface Trinket {

	/**
	 * Called every tick on the client and server side
	 *
	 * @param stack The stack being ticked
	 * @param slot The slot the stack is ticking in
	 * @param entity The entity wearing the stack
	 */
	default void tick(class_1799 stack, SlotReference slot, class_1309 entity) {
	}

	/**
	 * Called when an entity equips a trinket
	 *
	 * @param stack The stack being equipped
	 * @param slot The slot the stack is equipped to
	 * @param entity The entity that equipped the stack
	 */
	default void onEquip(class_1799 stack, SlotReference slot, class_1309 entity) {
	}

	/**
	 * Called when an entity equips a trinket
	 *
	 * @param stack The stack being unequipped
	 * @param slot The slot the stack was unequipped from
	 * @param entity The entity that unequipped the stack
	 */
	default void onUnequip(class_1799 stack, SlotReference slot, class_1309 entity) {
	}

	/**
	 * Determines whether an entity can equip a trinket
	 *
	 * @param stack The stack being equipped
	 * @param slot The slot the stack is being equipped to
	 * @param entity The entity that is equipping the stack
	 * @return Whether the stack can be equipped
	 */
	default boolean canEquip(class_1799 stack, SlotReference slot, class_1309 entity) {
		return true;
	}

	/**
	 * Determines whether an entity can unequip a trinket
	 *
	 * @param stack The stack being unequipped
	 * @param slot The slot the stack is being unequipped from
	 * @param entity The entity that is unequipping the stack
	 * @return Whether the stack can be unequipped
	 */
	default boolean canUnequip(class_1799 stack, SlotReference slot, class_1309 entity) {
		return !class_1890.method_60142(stack, class_9701.field_51656) || (entity instanceof class_1657 player && player.method_68878());
	}

	/**
	 * Determines whether a trinket can automatically attempt to equip into the first available
	 * slot when used
	 *
	 * @param stack The stack being equipped
	 * @param entity The entity that is using the stack
	 * @return Whether the stack can be equipped from use
	 */
	default boolean canEquipFromUse(class_1799 stack, class_1309 entity) {
		return false;
	}

	/**
	 * Determines the equip sound of a trinket
	 *
	 * @param stack The stack for the equip sound
	 * @param slot The slot the stack is being equipped to
	 * @param entity The entity that is equipping the stack
	 * @return The {@link class_3414} to play for equipping
	 */
	default class_6880<class_3414> getEquipSound(class_1799 stack, SlotReference slot, class_1309 entity) {
		class_10192 component = stack.method_58694(class_9334.field_54196);
		if (component != null) {
			return component.comp_3175();
		}
		return class_7923.field_41172.method_47983(class_3417.field_42593);
	}

	/**
	 * Returns the Entity Attribute Modifiers for a stack in a slot. Child implementations should
	 * remain pure
	 * <p>
	 * If modifiers do not change based on stack, slot, or entity, caching based on passed identifier
	 * should be considered
	 *
	 * @param stack ItemStack being polled for modifiers
	 * @param slot the {@link SlotReference} for the {@link TrinketInventory} the Trinket is relevant to
	 * @param entity the LivingEntity holding the Trinket
	 * @param slotIdentifier The Identifier to use for creating attributes
	 * @return the Multimap with any needed entries added
	 * @see SlotAttributes#addSlotModifier
	 */
	default Multimap<class_6880<class_1320>, class_1322> getModifiers(class_1799 stack, SlotReference slot, class_1309 entity, class_2960 slotIdentifier) {
		return Multimaps.newMultimap(Maps.newLinkedHashMap(), ArrayList::new);
	}

	/**
	 * Called by Trinkets when a trinket is broken on the client if {@link TrinketsApi#onTrinketBroken}
	 * is called by the callback in {@link class_1799#method_7956(int, class_3218, class_3222, Consumer)} server side
	 * <p>
	 * The default implementation works the same as breaking vanilla equipment, a sound is played and
	 * particles are spawned based on the item
	 *
	 * @param stack The stack being broken
	 * @param slot The slot the stack is being broken in
	 * @param entity The entity that is breaking the stack
	 */
	default void onBreak(class_1799 stack, SlotReference slot, class_1309 entity) {
		((LivingEntityAccessor) entity).invokePlayEquipmentBreakEffects(stack);
	}

	default TrinketEnums.DropRule getDropRule(class_1799 stack, SlotReference slot, class_1309 entity) {
		return TrinketEnums.DropRule.DEFAULT;
	}
}