package dev.emi.trinkets.api;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.mojang.serialization.Codec;
import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;
import net.minecraft.class_1262;
import net.minecraft.class_1263;
import net.minecraft.class_1309;
import net.minecraft.class_1322;
import net.minecraft.class_1657;
import net.minecraft.class_1799;
import net.minecraft.class_2371;
import net.minecraft.class_2960;
import net.minecraft.class_3218;

public class TrinketInventory implements class_1263 {
	private static final Codec<Collection<class_1322>> ENTITY_ATTRIBUTE_MODIFIERS_CODEC = class_1322.field_46247.listOf().xmap(Function.identity(), List::copyOf);

	private final SlotType slotType;
	private final int baseSize;
	private final TrinketComponent component;
	private final Map<class_2960, class_1322> modifiers = new HashMap<>();
	private final Set<class_1322> persistentModifiers = new HashSet<>();
	private final Set<class_1322> cachedModifiers = new HashSet<>();
	private final Multimap<class_1322.class_1323, class_1322> modifiersByOperation = HashMultimap.create();
	private final Consumer<TrinketInventory> updateCallback;

	private class_2371<class_1799> stacks;
	private boolean update = false;

	public TrinketInventory(SlotType slotType, TrinketComponent comp, Consumer<TrinketInventory> updateCallback) {
		this.component = comp;
		this.slotType = slotType;
		this.baseSize = slotType.getAmount();
		this.stacks = class_2371.method_10213(this.baseSize, class_1799.field_8037);
		this.updateCallback = updateCallback;
	}

	public SlotType getSlotType() {
		return this.slotType;
	}

	public TrinketComponent getComponent() {
		return this.component;
	}

	@Override
	public void method_5448() {
		for (int i = 0; i < this.method_5439(); i++) {
			stacks.set(i, class_1799.field_8037);
		}
	}

	@Override
	public int method_5439() {
		this.update();
		return this.stacks.size();
	}

	@Override
	public boolean method_5442() {
		for (int i = 0; i < this.method_5439(); i++) {
			if (!stacks.get(i).method_7960()) {
				return false;
			}
		}
		return true;
	}

	@Override
	public class_1799 method_5438(int slot) {
		this.update();
		return stacks.get(slot);
	}

	@Override
	public class_1799 method_5434(int slot, int amount) {
		return class_1262.method_5430(stacks, slot, amount);
	}

	@Override
	public class_1799 method_5441(int slot) {
		return class_1262.method_5428(stacks, slot);
	}

	@Override
	public void method_5447(int slot, class_1799 stack) {
		this.update();
		stacks.set(slot, stack);
	}

	@Override
	public void method_5431() {
		// NO-OP
	}

	public void markUpdate() {
		this.update = true;
		this.updateCallback.accept(this);
	}

	@Override
	public boolean method_5443(class_1657 player) {
		return true;
	}

	public Map<class_2960, class_1322> getModifiers() {
		return this.modifiers;
	}

	public Collection<class_1322> getModifiersByOperation(class_1322.class_1323 operation) {
		return this.modifiersByOperation.get(operation);
	}

	public void addModifier(class_1322 modifier) {
		this.modifiers.put(modifier.comp_2447(), modifier);
		this.getModifiersByOperation(modifier.comp_2450()).add(modifier);
		this.markUpdate();
	}

	public void addPersistentModifier(class_1322 modifier) {
		this.addModifier(modifier);
		this.persistentModifiers.add(modifier);
	}

	public void removeModifier(class_2960 identifier) {
		class_1322 modifier = this.modifiers.remove(identifier);
		if (modifier != null) {
			this.persistentModifiers.remove(modifier);
			this.getModifiersByOperation(modifier.comp_2450()).remove(modifier);
			this.markUpdate();
		}
	}

	public void clearModifiers() {
		java.util.Iterator<class_2960> iter = this.getModifiers().keySet().iterator();

		while(iter.hasNext()) {
			this.removeModifier(iter.next());
		}
	}

	public void removeCachedModifier(class_1322 attributeModifier) {
		this.cachedModifiers.remove(attributeModifier);
	}

	public void clearCachedModifiers() {
		for (class_1322 cachedModifier : this.cachedModifiers) {
			this.removeModifier(cachedModifier.comp_2447());
		}
		this.cachedModifiers.clear();
	}

	public void update() {
		if (this.update) {
			this.update = false;
			double baseSize = this.baseSize;
			for (class_1322 mod : this.getModifiersByOperation(class_1322.class_1323.field_6328)) {
				baseSize += mod.comp_2449();
			}

			double size = baseSize;
			for (class_1322 mod : this.getModifiersByOperation(class_1322.class_1323.field_6330)) {
				size += this.baseSize * mod.comp_2449();
			}

			for (class_1322 mod : this.getModifiersByOperation(class_1322.class_1323.field_6331)) {
				size *= mod.comp_2449();
			}
			class_1309 entity = this.component.getEntity();

			if (size != this.method_5439()) {
				class_2371<class_1799> newStacks = class_2371.method_10213((int) size, class_1799.field_8037);
				for (int i = 0; i < this.stacks.size(); i++) {
					class_1799 stack = this.stacks.get(i);
					if (i < newStacks.size()) {
						newStacks.set(i, stack);
					} else {
						if (entity.method_37908() instanceof class_3218 serverWorld) {
							entity.method_5775(serverWorld, stack);
						}
					}
				}

				this.stacks = newStacks;
			}
		}
	}

	public void copyFrom(TrinketInventory other) {
		this.modifiers.clear();
		this.modifiersByOperation.clear();
		this.persistentModifiers.clear();
		other.modifiers.forEach((uuid, modifier) -> this.addModifier(modifier));
		for (class_1322 persistentModifier : other.persistentModifiers) {
			this.addPersistentModifier(persistentModifier);
		}
		this.update();
	}

	public static void copyFrom(class_1309 previous, class_1309 current) {
		TrinketsApi.getTrinketComponent(previous).ifPresent(prevTrinkets -> {
			TrinketsApi.getTrinketComponent(current).ifPresent(currentTrinkets -> {
				Map<String, Map<String, TrinketInventory>> prevMap = prevTrinkets.getInventory();
				Map<String, Map<String, TrinketInventory>> currentMap = currentTrinkets.getInventory();
				for (Map.Entry<String, Map<String, TrinketInventory>> entry : prevMap.entrySet()) {
					Map<String, TrinketInventory> currentInvs = currentMap.get(entry.getKey());
					if (currentInvs != null) {
						for (Map.Entry<String, TrinketInventory> invEntry : entry.getValue().entrySet()) {
							TrinketInventory currentInv = currentInvs.get(invEntry.getKey());
							if (currentInv != null) {
								currentInv.copyFrom(invEntry.getValue());
							}
						}
					}
				}
			});
		});
	}

	public TrinketSaveData.Metadata toMetadata() {
		List<class_1322> cachedModifiers = new ArrayList<>();

		if (!this.modifiers.isEmpty()) {
			this.modifiers.forEach((uuid, modifier) -> {
				if (!this.persistentModifiers.contains(modifier)) {
					cachedModifiers.add(modifier);
				}
			});
		}
		return new TrinketSaveData.Metadata(List.copyOf(this.persistentModifiers), cachedModifiers);
	}

	public void fromMetadata(TrinketSaveData.Metadata tag) {
		tag.persistentModifiers().forEach(this::addPersistentModifier);

		if (!tag.cachedModifiers().isEmpty()) {
			for (class_1322 modifier : tag.cachedModifiers()) {
				this.cachedModifiers.add(modifier);
				this.addModifier(modifier);
			}

			this.update();
		}
	}

	public TrinketSaveData.Metadata getSyncMetadata() {
		return new TrinketSaveData.Metadata(List.copyOf(this.modifiers.values()), List.of());
	}

	public void applySyncMetadata(TrinketSaveData.Metadata metadata) {
		this.modifiers.clear();
		this.persistentModifiers.clear();
		this.modifiersByOperation.clear();

		metadata.persistentModifiers().forEach(this::addModifier);
		this.markUpdate();
		this.update();
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) return true;
		if (o == null || getClass() != o.getClass()) return false;
		TrinketInventory that = (TrinketInventory) o;
		return slotType.equals(that.slotType);
	}

	@Override
	public int hashCode() {
		return Objects.hash(slotType);
	}
}