package dev.emi.trinkets.mixin;

import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import dev.emi.trinkets.TrinketModifiers;
import dev.emi.trinkets.TrinketSlot;
import dev.emi.trinkets.api.SlotAttributes;
import dev.emi.trinkets.api.SlotReference;
import dev.emi.trinkets.api.SlotType;
import dev.emi.trinkets.api.TrinketInventory;
import dev.emi.trinkets.api.TrinketsApi;
import dev.emi.trinkets.api.TrinketsAttributeModifiersComponent;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.At.Shift;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfo;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import net.minecraft.class_10712;
import net.minecraft.class_124;
import net.minecraft.class_1320;
import net.minecraft.class_1322;
import net.minecraft.class_1657;
import net.minecraft.class_1792;
import net.minecraft.class_1799;
import net.minecraft.class_1836;
import net.minecraft.class_2561;
import net.minecraft.class_5134;
import net.minecraft.class_6880;
import net.minecraft.class_9285;

/**
 * Adds a tooltip for trinkets describing slots and attributes
 *
 * @author Emi
 */
@Mixin(class_1799.class)
public abstract class ItemStackMixin {


	@Inject(at = @At(value = "INVOKE", target = "Lnet/minecraft/item/ItemStack;appendAttributeModifiersTooltip(Ljava/util/function/Consumer;Lnet/minecraft/component/type/TooltipDisplayComponent;Lnet/minecraft/entity/player/PlayerEntity;)V", shift = Shift.BEFORE), method = "appendTooltip")
	private void getTooltip(class_1792.class_9635 context, class_10712 displayComponent, class_1657 player, class_1836 type, Consumer<class_2561> textConsumer, CallbackInfo ci) {
		TrinketsApi.getTrinketComponent(player).ifPresent(comp -> {
			class_1799 self = (class_1799) (Object) this;

            boolean showAttributeTooltip = displayComponent.method_67214(TrinketsAttributeModifiersComponent.TYPE);
			if (!showAttributeTooltip) {
				// nothing to do
				return;
			}

			boolean canEquipAnywhere = true;
			Set<SlotType> slots = Sets.newHashSet();
			Map<SlotType, Multimap<class_6880<class_1320>, class_1322>> modifiers = Maps.newHashMap();
			Multimap<class_6880<class_1320>, class_1322> defaultModifier = null;
			boolean allModifiersSame = true;
			int slotCount = 0;

			for (Map.Entry<String, Map<String, TrinketInventory>> group : comp.getInventory().entrySet()) {
				outer:
				for (Map.Entry<String, TrinketInventory> inventory : group.getValue().entrySet()) {
					TrinketInventory trinketInventory = inventory.getValue();
					SlotType slotType = trinketInventory.getSlotType();
					slotCount++;
					boolean anywhereButHidden = false;
					for (int i = 0; i < trinketInventory.method_5439(); i++) {
						SlotReference ref = new SlotReference(trinketInventory, i);
						boolean res = TrinketsApi.evaluatePredicateSet(slotType.getTooltipPredicates(), self, ref, player);
						boolean canInsert = TrinketSlot.canInsert(self, ref, player);
						if (res && canInsert) {
							boolean sameTranslationExists = false;
							for (SlotType t : slots) {
								if (t.getTranslation().getString().equals(slotType.getTranslation().getString())) {
									sameTranslationExists = true;
									break;
								}
							}
							if (!sameTranslationExists) {
								slots.add(slotType);
							}
							Multimap<class_6880<class_1320>, class_1322> map = TrinketModifiers.get(self, ref, player);

							if (defaultModifier == null) {
								defaultModifier = map;
							} else if (allModifiersSame) {
								allModifiersSame = areMapsEqual(defaultModifier, map);
							}

							boolean duplicate = false;
							for (Map.Entry<SlotType, Multimap<class_6880<class_1320>, class_1322>> entry : modifiers.entrySet()) {
								if (entry.getKey().getTranslation().getString().equals(slotType.getTranslation().getString())) {
									if (areMapsEqual(entry.getValue(), map)) {
										duplicate = true;
										break;
									}
								}
							}

							if (!duplicate) {
								modifiers.put(slotType, map);
							}
							continue outer;
						} else if (canInsert) {
							anywhereButHidden = true;
						}
					}
					if (!anywhereButHidden) {
						canEquipAnywhere = false;
					}
				}
			}

			if (canEquipAnywhere && slotCount > 1) {
				textConsumer.accept(class_2561.method_43471("trinkets.tooltip.slots.any").method_27692(class_124.field_1080));
			} else if (slots.size() > 1) {
				textConsumer.accept(class_2561.method_43471("trinkets.tooltip.slots.list").method_27692(class_124.field_1080));
				for (SlotType slotType : slots) {
					textConsumer.accept(slotType.getTranslation().method_27692(class_124.field_1078));
				}
			} else if (slots.size() == 1) {
				// Should only run once
				for (SlotType slotType : slots) {
					textConsumer.accept(class_2561.method_43469("trinkets.tooltip.slots.single",
								slotType.getTranslation().method_27692(class_124.field_1078)).method_27692(class_124.field_1080));
				}
			}

			if (!modifiers.isEmpty() && showAttributeTooltip) {
				if (allModifiersSame) {
					if (defaultModifier != null && !defaultModifier.isEmpty()) {
						textConsumer.accept(class_2561.method_43471("trinkets.tooltip.attributes.all").method_27692(class_124.field_1080));
						addAttributes(textConsumer, defaultModifier);
					}
				} else {
					for (Map.Entry<SlotType, Multimap<class_6880<class_1320>, class_1322>> entry : modifiers.entrySet()) {
						textConsumer.accept(class_2561.method_43469("trinkets.tooltip.attributes.single",
								entry.getKey().getTranslation().method_27692(class_124.field_1078)).method_27692(class_124.field_1080));
						addAttributes(textConsumer, entry.getValue());
					}
				}
			}
		});
	}

	@Unique
	private void addAttributes(Consumer<class_2561> textConsumer, Multimap<class_6880<class_1320>, class_1322> map) {
		if (!map.isEmpty()) {
			for (Map.Entry<class_6880<class_1320>, class_1322> entry : map.entries()) {
				class_6880<class_1320> attribute = entry.getKey();
				class_1322 modifier = entry.getValue();
				double g = modifier.comp_2449();

				if (modifier.comp_2450() != class_1322.class_1323.field_6330 && modifier.comp_2450() != class_1322.class_1323.field_6331) {
					if (entry.getKey().equals(class_5134.field_23718)) {
						g *= 10.0D;
					}
				} else {
					g *= 100.0D;
				}

				class_2561 text = class_2561.method_43471(attribute.comp_349().method_26830());
				if (attribute.method_40227() && attribute.comp_349() instanceof SlotAttributes.SlotEntityAttribute) {
					text = class_2561.method_43469("trinkets.tooltip.attributes.slots", text);
				}
				if (g > 0.0D) {
					textConsumer.accept(class_2561.method_43469("attribute.modifier.plus." + modifier.comp_2450().method_56082(),
							class_9285.field_49329.format(g), text).method_27692(class_124.field_1078));
				} else if (g < 0.0D) {
					g *= -1.0D;
					textConsumer.accept(class_2561.method_43469("attribute.modifier.take." + modifier.comp_2450().method_56082(),
							class_9285.field_49329.format(g), text).method_27692(class_124.field_1061));
				}
			}
		}
	}

	// `equals` doesn't test thoroughly
	@Unique
	private boolean areMapsEqual(Multimap<class_6880<class_1320>, class_1322> map1, Multimap<class_6880<class_1320>, class_1322> map2) {
		if (map1.size() != map2.size()) {
			return false;
		} else {
			for (class_6880<class_1320> attribute : map1.keySet()) {
				if (!map2.containsKey(attribute)) {
					return false;
				}

				Collection<class_1322> col1 = map1.get(attribute);
				Collection<class_1322> col2 = map2.get(attribute);

				if (col1.size() != col2.size()) {
					return false;
				} else {
					Iterator<class_1322> iter = col2.iterator();

					for (class_1322 modifier : col1) {
						class_1322 eam = iter.next();

						//we can't check identifiers. EAMs will have slot-specific identifiers so fail total equality by nature.
						if (!modifier.comp_2450().equals(eam.comp_2450())) {
							return false;
						}
						if (modifier.comp_2449() != eam.comp_2449()) {
							return false;
						}
					}
				}
			}
		}
		return true;
	}
}