package eu.pb4.mapcanvas.impl.font;

import eu.pb4.mapcanvas.api.core.CanvasColor;
import eu.pb4.mapcanvas.api.core.DrawableCanvas;
import eu.pb4.mapcanvas.api.font.CanvasFont;
import eu.pb4.mapcanvas.api.font.FontUtils;
import it.unimi.dsi.fastutil.ints.Int2ObjectMap;
import it.unimi.dsi.fastutil.ints.Int2ObjectOpenHashMap;
import net.minecraft.util.math.MathHelper;
import org.jetbrains.annotations.ApiStatus;

import java.util.Arrays;

public final class BitmapFont implements CanvasFont {
    public static final BitmapFont EMPTY = new BitmapFont(Glyph.INVALID, Metadata.empty());

    public final Int2ObjectMap<Glyph> characters = new Int2ObjectOpenHashMap<>();
    public final Glyph defaultGlyph;
    private final Metadata metadata;

    public BitmapFont(Glyph defaultGlyph, Metadata metadata) {
        this.defaultGlyph = defaultGlyph;
        this.metadata = metadata;
    }

    public BitmapFont(Glyph defaultGlyph, Int2ObjectMap<Glyph> characters, Metadata metadata) {
        this.defaultGlyph = defaultGlyph;

        for (var entry : characters.int2ObjectEntrySet()) {
            this.characters.put(entry.getIntKey(), entry.getValue());
        }

        this.metadata = metadata;
    }

    public BitmapFont(Glyph defaultGlyph, Int2ObjectMap<Glyph> characters) {
        this(defaultGlyph, characters, Metadata.empty());
    }

    @Override
    public int getGlyphWidth(int character, double size, int offset) {
        return this.characters.getOrDefault(character, this.defaultGlyph).getWidth(size, offset);
    }

    @Override
    public int drawGlyph(DrawableCanvas canvas, int character, int x, int y, double size, int offset, CanvasColor color) {
        return this.characters.getOrDefault(character, this.defaultGlyph).draw(canvas, x, y, size, offset, color);
    }

    @Override
    public boolean containsGlyph(int character) {
        return this.characters.containsKey(character);
    }

    @Override
    public Metadata getMetadata() {
        return this.metadata;
    }

    public record Glyph(int width, int height, int ascend, int fontWidth, int logicalHeight, boolean[] texture) {
        public static final Glyph INVALID;
        public static final Glyph EMPTY = new Glyph(0, 0, 0, 0, 0, new boolean[]{});
        public static final Glyph ATLAS = new Glyph(8, 8, 8, 8, 8, convert(8 * 8,
                """
                        xxxxxxxx
                        xx----xx
                        x-x--x-x
                        x--xx--x
                        x--xx--x
                        x-x--x-x
                        xx----xx
                        xxxxxxxx
                        """
        ));

        public static final Glyph PLAYER = new Glyph(8, 8, 8, 8, 8, convert(8 * 8,
                """
                        xxxxxxxx
                        xxxxxxxx
                        x------x
                        --------
                        -xx--xx-
                        ---xx---
                        --x--x--
                        --xxxx--
                        """
        ));

        static {
            var array = new boolean[8 * 5];

            for (int x = 0; x < 5; x++) {
                for (int y = 0; y < 8; y++) {
                    if (x == 0 || y == 0 || x == 4 || y == 7) {
                        array[x + y * 5] = true;
                    }
                }
            }

            INVALID = new Glyph(5, 8, 7, 4, 8, array);
        }

        private static boolean[] convert(int size, String string) {
            var texture = new boolean[size];
            string = string.replace("\n", "");
            for (var i = 0; i < string.length(); i++) {
                texture[i] = string.charAt(i) == 'x';
            }
            return texture;
        }

        public int draw(DrawableCanvas canvas, int x, int y, double size, int offset, CanvasColor color) {
            if (this.logicalHeight() == 0 || this.height() == 0) {
                return (int) (((this.fontWidth())) * (size / 8));
            }

            final double textureScale = (double) this.height() / this.logicalHeight();
            final double baseScale = size / textureScale / 8;

            for (int fX = 0; fX < this.width(); fX++) {
                for (int fY = 0; fY < this.height(); fY++) {
                    if (this.texture()[fX + fY * this.width()]) {
                        for (int lX = 0; lX < baseScale; lX++) {
                            for (int lY = 0; lY < baseScale; lY++) {
                                canvas.set(
                                        x + MathHelper.floor(fX * baseScale) + lX,
                                        y + MathHelper.floor((fY + (7 - this.ascend()) * textureScale) * baseScale) + lY,
                                        color
                                );
                            }
                        }
                    }
                }
            }

            return (int) (((this.fontWidth() + offset)) * baseScale);
        }

        public int getWidth(double size, int offset) {
            if (this.logicalHeight() == 0 || this.height() == 0) {
                return (int) (((this.fontWidth())) * (size / 8));
            }

            final double textureScale = (double) this.height() / this.logicalHeight();
            final double baseScale = size / textureScale / 8;

            return (int) (((this.fontWidth() + offset)) * baseScale);
        }
    }

    public static BitmapFont tryConvert(CanvasFont font) {
        return switch (font) {
            case BitmapFont bitmapFont -> bitmapFont;
            case LazyFont lazyFont -> tryConvert(lazyFont.font());
            case StackedFont stackedFont when FontUtils.isBitmapFont(stackedFont) -> {
                var fonts = Arrays.stream(stackedFont.fonts()).map(BitmapFont::tryConvert).toArray(BitmapFont[]::new);
                var out = new BitmapFont(fonts[0].defaultGlyph, stackedFont.getMetadata());
                for (var i = fonts.length - 1; i >= 0; i--) {
                    out.characters.putAll(fonts[i].characters);
                }

                yield out;
            }
            case StackedLazyFont stackedFont when FontUtils.isBitmapFont(stackedFont) -> {
                var fonts = Arrays.stream(stackedFont.fonts()).map(BitmapFont::tryConvert).toArray(BitmapFont[]::new);
                var out = new BitmapFont(fonts[0].defaultGlyph, stackedFont.getMetadata());
                for (var i = fonts.length - 1; i >= 0; i--) {
                    out.characters.putAll(fonts[i].characters);
                }

                yield out;
            }
            default -> null;
        };
    }
}
