package eu.pb4.sgui.api.elements;

import com.google.common.collect.ImmutableMultimap;
import com.mojang.authlib.GameProfile;
import com.mojang.authlib.minecraft.MinecraftProfileTextures;
import com.mojang.authlib.properties.Property;
import com.mojang.authlib.properties.PropertyMap;
import com.mojang.datafixers.util.Either;
import eu.pb4.sgui.api.GuiHelpers;
import eu.pb4.sgui.mixin.StaticAccessor;
import it.unimi.dsi.fastutil.objects.ReferenceSortedSets;
import net.minecraft.component.ComponentType;
import net.minecraft.component.DataComponentTypes;
import net.minecraft.component.type.CustomModelDataComponent;
import net.minecraft.component.type.LoreComponent;
import net.minecraft.component.type.ProfileComponent;
import net.minecraft.component.type.TooltipDisplayComponent;
import net.minecraft.enchantment.Enchantment;
import net.minecraft.entity.player.SkinTextures;
import net.minecraft.item.Item;
import net.minecraft.item.ItemStack;
import net.minecraft.item.Items;
import net.minecraft.item.tooltip.TooltipAppender;
import net.minecraft.registry.RegistryKey;
import net.minecraft.registry.RegistryKeys;
import net.minecraft.registry.RegistryWrapper;
import net.minecraft.registry.entry.RegistryEntry;
import net.minecraft.server.MinecraftServer;
import net.minecraft.text.Text;
import net.minecraft.util.*;
import org.jetbrains.annotations.Nullable;

import java.util.*;

/**
 * Gui Element Builder
 * <br>
 * The GuiElementBuilder is the best way of constructing gui elements.
 * It supplies all the methods needed to construct a standard {@link GuiElement}.
 *
 * @see GuiElementBuilderInterface
 */
@SuppressWarnings({"unused"})
public class GuiElementBuilder implements GuiElementBuilderInterface<GuiElementBuilder> {
    protected ItemStack itemStack = new ItemStack(Items.WHITE_DYE);
    protected GuiElement.ClickCallback callback = GuiElementInterface.EMPTY_CALLBACK;
    private boolean hideComponentTooltips;
    private boolean noTooltips;

    /**
     * Constructs a GuiElementBuilder with the default options
     */
    public GuiElementBuilder() {
    }

    /**
     * Constructs a GuiElementBuilder with the specified Item.
     *
     * @param item the item to use
     */
    public GuiElementBuilder(Item item) {
        this.itemStack = new ItemStack(item);
    }

    /**
     * Constructs a GuiElementBuilder with the specified item model.
     *
     * @param model Item model to use. Same as calling model(...).
     */
    public GuiElementBuilder(Identifier model) {
        this.model(model);
    }

    /**
     * Constructs a GuiElementBuilder with the specified Item
     * and number of items.
     *
     * @param item  the item to use
     * @param count the number of items
     */
    public GuiElementBuilder(Item item, int count) {
        this.itemStack = new ItemStack(item, count);
    }

    /**
     * Constructs a GuiElementBuilder with the specified ItemStack
     *
     * @param stack  the item stack to use
     */
    public GuiElementBuilder(ItemStack stack) {
        this.itemStack = stack.copy();
    }

    /**
     * Constructs a GuiElementBuilder based on the supplied stack.
     *
     * @param stack the stack to base the builder of
     * @return the constructed builder
     */
    public static GuiElementBuilder from(ItemStack stack) {
        return new GuiElementBuilder(stack);
    }

    @Deprecated
    public static List<Text> getLore(ItemStack stack) {
        return stack.getOrDefault(DataComponentTypes.LORE, LoreComponent.DEFAULT).lines();
    }

    /**
     * Sets the type of Item of the element.
     *
     * @param item the item to use
     * @return this element builder
     */
    public GuiElementBuilder setItem(Item item) {
        this.itemStack = new ItemStack(item.getRegistryEntry(), this.itemStack.getCount(), this.itemStack.getComponentChanges());
        return this;
    }

    /**
     * Sets the name of the element.
     *
     * @param name the name to use
     * @return this element builder
     */
    public GuiElementBuilder setName(Text name) {
        this.itemStack.set(DataComponentTypes.CUSTOM_NAME, name.copy().styled(GuiHelpers.STYLE_CLEARER));
        return this;
    }

    /**
     * Sets the item name of the element.
     *
     * @param name the name to use
     * @return this element builder
     */
    public GuiElementBuilder setItemName(Text name) {
        this.itemStack.set(DataComponentTypes.ITEM_NAME, name.copy());
        return this;
    }

    /**
     * Sets the rarity of the element.
     *
     * @param rarity to use
     * @return this element builder
     */
    public GuiElementBuilder setRarity(Rarity rarity) {
        this.itemStack.set(DataComponentTypes.RARITY, rarity);
        return this;
    }

    /**
     * Sets the number of items in the element.
     *
     * @param count the number of items
     * @return this element builder
     */
    public GuiElementBuilder setCount(int count) {
        this.itemStack.setCount(count);
        return this;
    }


    /**
     * Sets the max number of items in the element.
     *
     * @param count the number of items
     * @return this element builder
     */
    public GuiElementBuilder setMaxCount(int count) {
        this.itemStack.set(DataComponentTypes.MAX_STACK_SIZE, count);
        return this;
    }

    /**
     * Sets the lore lines of the element.
     *
     * @param lore a list of all the lore lines
     * @return this element builder
     */
    public GuiElementBuilder setLore(List<Text> lore) {
        var l = new ArrayList<Text>(lore.size());
        for (var t : lore) {
            l.add(t.copy().styled(GuiHelpers.STYLE_CLEARER));
        }

        this.itemStack.set(DataComponentTypes.LORE, new LoreComponent(l));
        return this;
    }

    /**
     * Sets the lore lines of the element, without clearing out formatting.
     *
     * @param lore a list of all the lore lines
     * @return this element builder
     */
    public GuiElementBuilder setLoreRaw(List<Text> lore) {
        this.itemStack.set(DataComponentTypes.LORE, new LoreComponent(lore));
        return this;
    }

    /**
     * Adds a line of lore to the element.
     *
     * @param lore the line to add
     * @return this element builder
     */
    public GuiElementBuilder addLoreLine(Text lore) {
        this.itemStack.apply(DataComponentTypes.LORE, LoreComponent.DEFAULT, lore.copy().styled(GuiHelpers.STYLE_CLEARER), LoreComponent::with);
        return this;
    }

    /**
     * Adds a line of lore to the element, without clearing out formatting.
     *
     * @param lore the line to add
     * @return this element builder
     */
    public GuiElementBuilder addLoreLineRaw(Text lore) {
        this.itemStack.apply(DataComponentTypes.LORE, LoreComponent.DEFAULT, lore, LoreComponent::with);
        return this;
    }

    /**
     * Set the damage of the element. This will only be
     * visible if the item supports has durability.
     *
     * @param damage the amount of durability the item is missing
     * @return this element builder
     */
    public GuiElementBuilder setDamage(int damage) {
        this.itemStack.set(DataComponentTypes.DAMAGE, damage);
        return this;
    }

    /**
     * Set the max damage of the element.
     *
     * @param damage the amount of durability the item is missing
     * @return this element builder
     */
    public GuiElementBuilder setMaxDamage(int damage) {
        this.itemStack.set(DataComponentTypes.MAX_DAMAGE, damage);
        return this;
    }

    /**
     * Disables all default components on an item.
     * @return this element builder
     */
    public GuiElementBuilder noDefaults() {
        for (var x : this.itemStack.getItem().getComponents()) {
            if (x.type() == DataComponentTypes.ITEM_MODEL) {
                continue;
            }
            if (this.itemStack.get(x.type()) == x.value()) {
                this.itemStack.set(x.type(), null);
            }
        }
        return this;
    }

    @Nullable
    public <T> T getComponent(ComponentType<T> type) {
        return this.itemStack.get(type);
    }

    public <T> GuiElementBuilder setComponent(ComponentType<T> type, @Nullable T value) {
        this.itemStack.set(type, value);
        return this;
    }

    /**
     * Hides all component-item related tooltip added by item's or non name/lore components.
     *
     * @return this element builder
     */
    public GuiElementBuilder hideDefaultTooltip() {
        this.hideComponentTooltips = true;
        return this;
    }

    /**
     * Hides tooltip completely, making it never show
     * @return this element builder
     */
    public GuiElementBuilder hideTooltip() {
        this.noTooltips = true;
        return this;
    }

    /**
     * Give the element the specified enchantment.
     *
     * @param enchantment the enchantment to apply
     * @param level       the level of the specified enchantment
     * @return this element builder
     */
    public GuiElementBuilder enchant(RegistryEntry<Enchantment> enchantment, int level) {
        this.itemStack.addEnchantment(enchantment, level);
        return this;
    }

    /**
     * Give the element the specified enchantment.
     *
     * @param server MinecraftServer
     * @param enchantment the enchantment to apply
     * @param level       the level of the specified enchantment
     * @return this element builder
     */
    public GuiElementBuilder enchant(MinecraftServer server, RegistryKey<Enchantment> enchantment, int level) {
        return enchant(server.getRegistryManager(), enchantment, level);
    }

    /**
     * Give the element the specified enchantment.
     *
     * @param lookup WrapperLookup
     * @param enchantment the enchantment to apply
     * @param level       the level of the specified enchantment
     * @return this element builder
     */
    public GuiElementBuilder enchant(RegistryWrapper.WrapperLookup lookup, RegistryKey<Enchantment> enchantment, int level) {
        return enchant(lookup.getOrThrow(RegistryKeys.ENCHANTMENT).getOrThrow(enchantment), level);
    }

    /**
     * Sets the element to have an enchantment glint.
     *
     * @return this element builder
     */
    public GuiElementBuilder glow() {
        this.itemStack.set(DataComponentTypes.ENCHANTMENT_GLINT_OVERRIDE, true);
        return this;
    }

    /**
     * Sets the element to have an enchantment glint.
     *
     * @return this element builder
     */
    public GuiElementBuilder glow(boolean value) {
        this.itemStack.set(DataComponentTypes.ENCHANTMENT_GLINT_OVERRIDE, value);
        return this;
    }

    /**
     * Sets the custom model data of the element.
     *
     * @return this element builder
     */
    public GuiElementBuilder setCustomModelData(List<Float> floats, List<Boolean> flags, List<String> strings, List<Integer> colors) {
        this.itemStack.set(DataComponentTypes.CUSTOM_MODEL_DATA, new CustomModelDataComponent(floats, flags, strings, colors));
        return this;
    }

    /**
     * Sets the model of the element.
     *
     * @param model model to display item as
     * @return this element builder
     */
    public GuiElementBuilder model(Identifier model) {
        this.itemStack.set(DataComponentTypes.ITEM_MODEL, model);
        return this;
    }

    public GuiElementBuilder model(Item model) {
        this.itemStack.set(DataComponentTypes.ITEM_MODEL, model.getComponents().get(DataComponentTypes.ITEM_MODEL));
        return this;
    }

    /**
     * Sets the element to be unbreakable, also hides the durability bar.
     *
     * @return this element builder
     */
    public GuiElementBuilder unbreakable() {
        this.itemStack.set(DataComponentTypes.UNBREAKABLE, Unit.INSTANCE);
        return this;
    }

    /**
     * Sets the skull owner tag of a player head.
     * If the server parameter is not supplied it may lag the client while it loads the texture,
     * otherwise if the server is provided and the {@link GameProfile} contains a UUID then the
     * textures will be loaded by the server. This can take some time the first load,
     * however the skins are cached for later uses so its often less noticeable to let the
     * server load the textures.
     *
     * @param profile the {@link GameProfile} of the owner
     * @return this element builder
     */
    public GuiElementBuilder setProfile(GameProfile profile) {
        if (!profile.properties().isEmpty()) {
            return this.setProfile(ProfileComponent.ofStatic(profile));
        }
        if (profile.name().isEmpty()) {
            return this.setProfile(ProfileComponent.ofDynamic(profile.id()));
        }
        if (profile.id().equals(Util.NIL_UUID)) {
            return this.setProfile(ProfileComponent.ofDynamic(profile.name()));
        }
        return this;
    }

    public GuiElementBuilder setProfile(String name) {
        return this.setProfile(ProfileComponent.ofDynamic(name));
    }

    public GuiElementBuilder setProfile(UUID uuid) {
        return this.setProfile(ProfileComponent.ofDynamic(uuid));
    }

    public GuiElementBuilder setProfile(Identifier textureId) {
        return this.setProfile(StaticAccessor.createStatic(Either.right(ProfileComponent.Data.EMPTY),
                new SkinTextures.SkinOverride(Optional.of(new AssetInfo.TextureAssetInfo(textureId)), Optional.empty(),
                        Optional.empty(), Optional.empty())));
    }

    public GuiElementBuilder setProfile(SkinTextures.SkinOverride info) {
        return this.setProfile(StaticAccessor.createStatic(Either.right(ProfileComponent.Data.EMPTY), info));
    }

    public GuiElementBuilder setProfile(ProfileComponent component) {
        this.itemStack.set(DataComponentTypes.PROFILE, component);
        return this;
    }


    public GuiElementBuilder setProfileSkinTexture(String value) {
        return this.setProfileSkinTexture(value, null, null);
    }

    public GuiElementBuilder setProfileSkinTexture(String value, @Nullable String signature, @Nullable UUID uuid) {
        PropertyMap map = new PropertyMap(ImmutableMultimap.of("textures", new Property("textures", value, signature)));
        return this.setProfile(new GameProfile( uuid != null ? uuid : Util.NIL_UUID, "", map));
    }

    /**
     * Sets the skull owner tag of a player head.
     * If the server parameter is not supplied it may lag the client while it loads the texture,
     * otherwise if the server is provided and the {@link GameProfile} contains a UUID then the
     * textures will be loaded by the server. This can take some time the first load,
     * however the skins are cached for later uses so its often less noticeable to let the
     * server load the textures.
     *
     * @param profile the {@link GameProfile} of the owner
     * @return this element builder
     */
    @Deprecated
    public GuiElementBuilder setSkullOwner(GameProfile profile, @Nullable MinecraftServer server) {
        return this.setProfile(profile);
    }

    /**
     * Sets the skull owner tag of a player head.
     * This method uses raw values required by client to display the skin
     * Ideal for textures generated with 3rd party websites like mineskin.org
     *
     * @param value     texture value used by client
     * @return this element builder
     */
    @Deprecated
    public GuiElementBuilder setSkullOwner(String value) {
        return this.setSkullOwner(value, null, null);
    }

    /**
     * Sets the skull owner tag of a player head.
     * This method uses raw values required by client to display the skin
     * Ideal for textures generated with 3rd party websites like mineskin.org
     *
     * @param value     texture value used by client
     * @param signature optional signature, will be ignored when set to null
     * @param uuid      UUID of skin owner, if null default will be used
     * @return this element builder
     */
    @Deprecated
    public GuiElementBuilder setSkullOwner(String value, @Nullable String signature, @Nullable UUID uuid) {
        return this.setProfileSkinTexture(value, signature, uuid);
    }
    
    @Override
    public GuiElementBuilder setCallback(GuiElement.ClickCallback callback) {
        this.callback = callback;
        return this;
    }

    @Override
    public GuiElementBuilder setCallback(GuiElementInterface.ItemClickCallback callback) {
        this.callback = callback;
        return this;
    }

    /**
     * Constructs an ItemStack using the current builder options.
     * Note that this ignores the callback as it is stored in
     * the {@link GuiElement}.
     *
     * @return this builder as a stack
     * @see GuiElementBuilder#build()
     */
    public ItemStack asStack() {
        var copy = itemStack.copy();
        if (this.noTooltips) {
            copy.set(DataComponentTypes.TOOLTIP_DISPLAY, new TooltipDisplayComponent(true, ReferenceSortedSets.emptySet()));
        } else if (this.hideComponentTooltips) {
            var comp = TooltipDisplayComponent.DEFAULT;
            for (var entry : this.itemStack.getComponents()) {
                if (entry.type() != DataComponentTypes.ITEM_NAME && entry.type() != DataComponentTypes.CUSTOM_NAME && entry.type() != DataComponentTypes.LORE) {
                    comp = comp.with(entry.type(), true);
                }
            }
            copy.set(DataComponentTypes.TOOLTIP_DISPLAY, comp);
        }

        return copy;
    }

    @Override
    public GuiElement build() {
        return new GuiElement(this.asStack(), this.callback);
    }

    @Deprecated(forRemoval = true)
    public GuiElementBuilder hideFlags() {
        return this.hideDefaultTooltip();
    }
}
