package eu.pb4.sgui.api.gui;

import eu.pb4.sgui.api.ScreenProperty;
import eu.pb4.sgui.api.elements.BookElementBuilder;
import eu.pb4.sgui.virtual.SguiScreenHandlerFactory;
import eu.pb4.sgui.virtual.book.BookScreenHandler;
import net.minecraft.item.ItemStack;
import net.minecraft.registry.tag.ItemTags;
import net.minecraft.screen.ScreenHandlerType;
import net.minecraft.server.network.ServerPlayerEntity;
import net.minecraft.text.Text;

import java.util.OptionalInt;

/**
 * Book Gui Implementation
 * <p>
 * BookGui is used to display book pages to the player. A pre-existing book needs
 * to be passed into the constructor, this is what will be displayed.
 * <p>
 * BookGui has lots of deprecated methods which have no function, this is
 * mainly due to the lack of item slots in the book interface.
 */
@SuppressWarnings("unused")
public class BookGui implements GuiInterface {
    protected final ServerPlayerEntity player;
    protected ItemStack book;
    protected int page = 0;
    @Deprecated(forRemoval = true)
    protected boolean open = false;
    protected boolean reOpen = false;
    protected BookScreenHandler screenHandler = null;

    protected int syncId = -1;

    /**
     * Constructs a new BookGui for the supplied player, based
     * on the provided book.
     *
     * @param player the player to serve this gui to
     * @param book   the book stack to display
     * @throws IllegalArgumentException if the provided item is not a book
     */
    public BookGui(ServerPlayerEntity player, ItemStack book) {
        this.player = player;
        this.book = book;
    }

    public BookGui(ServerPlayerEntity player) {
        this(player, ItemStack.EMPTY);
    }

    /**
     * Constructs a new BookGui for the supplied player, based
     * on the provided book.
     *
     * @param player the player to serve this gui to
     * @param book   the book builder to display
     */
    public BookGui(ServerPlayerEntity player, BookElementBuilder book) {
        this.player = player;
        this.book = book.asStack();
    }

    /**
     * Sets the selected page number
     *
     * @param page the page index, from 0
     */
    public void setPage(int page) {
        this.page = page;
        this.sendProperty(ScreenProperty.SELECTED, this.page);
    }

    /**
     * Returns the current selected page
     *
     * @return the page index, from 0
     */
    public int getPage() {
        return page;
    }

    /**
     * Returns the book item used to store the data.
     *
     * @return the book stack
     */
    public ItemStack getBook() {
        return this.book;
    }

    /**
     * Updates the book.
     */
    public void setBook(ItemStack book) {
        this.book = book;
    }

    /**
     * Updates the book.
     */
    public void setBook(BookElementBuilder book) {
        this.book = book.asStack();
    }

    /**
     * Activates when the 'Take Book' button is pressed
     */
    public void onTakeBookButton() {
    }

    /**
     * Activates when player selects a page (via click event).
     * The page passed is equal to one provided in click event!
     */
    public void onPageSelected(int page) {
        this.setPage(page - 1);
    }

    /**
     * Activates when player selects a page (via click event).
     */
    public void onNextPageButton() {
        this.setPage(this.getPage() + 1);
    }

    /**
     * Activates when player selects a page (via click event).
     */
    public void onPreviousPageButton() {
        this.setPage(this.getPage() - 1);
    }

    @Override
    public ScreenHandlerType<?> getType() { return ScreenHandlerType.LECTERN; }

    @Override
    public ServerPlayerEntity getPlayer() {
        return this.player;
    }

    @Override
    public int getSyncId() {
        return this.syncId;
    }

    @Override
    public boolean isOpen() {
        return this.screenHandler == this.player.currentScreenHandler;
    }

    @Override
    public boolean open() {
        var state = false;
        if (!this.player.isDisconnected() && !this.isOpen()) {
            this.beforeOpen();
            state = this.setupScreenHandler();
            this.afterOpen();
        }
        return state;
    }

    protected boolean setupScreenHandler() {
        //noinspection removal
        this.open = true;
        this.onOpen();
        this.reOpen = true;
        OptionalInt temp = this.player.openHandledScreen(new SguiScreenHandlerFactory<>(this, (syncId, inv, player) -> new BookScreenHandler(syncId, this, player)));
        this.reOpen = false;
        if (temp.isPresent()) {
            this.syncId = temp.getAsInt();
            if (this.player.currentScreenHandler instanceof BookScreenHandler) {
                this.screenHandler = (BookScreenHandler) this.player.currentScreenHandler;
                this.sendProperty(ScreenProperty.SELECTED, this.page);
                return true;
            }
        }
        return false;
    }

    /**
     * Called when player executes command via {@link net.minecraft.text.ClickEvent.Action#RUN_COMMAND}
     *
     * @param command input command
     * @return Returns false, for continuing execution or true, if you want to cancel it
     */
    public boolean onCommand(String command) {
        return false;
    }

    @Override
    public void close(boolean screenHandlerIsClosed) {
        if (this.isOpen() && !this.reOpen) {
            //noinspection removal
            this.open = this.isOpen();
            this.reOpen = false;

            if (!screenHandlerIsClosed && this.player.currentScreenHandler == this.screenHandler) {
                this.player.closeHandledScreen();
            }

            this.onClose();
        } else {
            this.reOpen = false;
        }
    }

    @Deprecated
    @Override
    public void setTitle(Text title) {
    }

    @Deprecated
    @Override
    public Text getTitle() {
        return null;
    }

    @Deprecated
    @Override
    public boolean getAutoUpdate() {
        return false;
    }

    @Deprecated
    @Override
    public void setAutoUpdate(boolean value) {
    }
}
