package eu.pb4.sgui.api.gui;

import eu.pb4.sgui.api.ClickType;
import eu.pb4.sgui.api.GuiHelpers;
import eu.pb4.sgui.api.elements.GuiElementInterface;
import eu.pb4.sgui.mixin.ServerPlayerEntityAccessor;
import eu.pb4.sgui.virtual.hotbar.HotbarScreenHandler;
import eu.pb4.sgui.virtual.inventory.VirtualSlot;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.ints.IntList;
import net.minecraft.item.ItemStack;
import net.minecraft.network.packet.c2s.play.PlayerActionC2SPacket;
import net.minecraft.network.packet.s2c.play.*;
import net.minecraft.screen.ScreenHandler;
import net.minecraft.screen.ScreenHandlerSyncHandler;
import net.minecraft.screen.ScreenHandlerType;
import net.minecraft.screen.slot.Slot;
import net.minecraft.screen.slot.SlotActionType;
import net.minecraft.server.network.ServerPlayerEntity;
import net.minecraft.text.Text;
import net.minecraft.util.collection.DefaultedList;
import net.minecraft.util.hit.BlockHitResult;
import net.minecraft.util.math.BlockPos;
import net.minecraft.util.math.Direction;
import net.minecraft.util.math.MathHelper;
import net.minecraft.util.math.Vec3d;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Nullable;

/**
 * It's a gui implementation for hotbar/player inventory usage
 * <p>
 * Unlike other Slot based guis, it doesn't extend SimpleGui
 */
public class HotbarGui extends BaseSlotGui {
    public static final int SIZE = 46;
    public static final int[] VANILLA_HOTBAR_SLOT_IDS = createArrayFromTo(36, 44);
    public static final int[] VANILLA_BACKPACK_SLOT_IDS = createArrayFromTo(9, 35);
    public static final int[] VANILLA_ARMOR_SLOT_IDS = createArrayFromTo(5, 8);
    public static final int VANILLA_OFFHAND_SLOT_ID = 45;
    public static final int[] VANILLA_CRAFTING_IDS = new int[]{1, 2, 3, 4, 0};
    public static final int[] GUI_TO_VANILLA_IDS = mergeArrays(VANILLA_HOTBAR_SLOT_IDS, new int[]{VANILLA_OFFHAND_SLOT_ID}, VANILLA_BACKPACK_SLOT_IDS, VANILLA_ARMOR_SLOT_IDS, VANILLA_CRAFTING_IDS);
    public static final int[] VANILLA_TO_GUI_IDS = rotateArray(GUI_TO_VANILLA_IDS);
    protected int selectedSlot = 0;
    protected boolean hasRedirects = false;
    private HotbarScreenHandler screenHandler;
    private int clicksPerTick;

    public HotbarGui(ServerPlayerEntity player) {
        super(player, SIZE);
    }

    private static int[] rotateArray(int[] input) {
        int[] array = new int[input.length];
        for (int i = 0; i < array.length; i++) {
            array[input[i]] = i;
        }
        return array;
    }

    private static int[] createArrayFromTo(int first, int last) {
        IntList list = new IntArrayList(last - first);
        for (int i = first; i <= last; i++) {
            list.add(i);
        }

        return list.toIntArray();
    }

    private static int[] mergeArrays(int[]... idArrays) {
        IntList list = new IntArrayList(SIZE);
        for (var array : idArrays) {
            for (int i : array) {
                list.add(i);
            }
        }
        return list.toIntArray();
    }

    @Override
    public void setSlot(int index, GuiElementInterface element) {
        super.setSlot(index, element);
        if (this.isOpen() && this.autoUpdate) {
            this.screenHandler.setSlot(GUI_TO_VANILLA_IDS[index], new VirtualSlot(this, index, 0, 0));
        }
    }

    public void setSlotRedirect(int index, Slot slot) {
        super.setSlotRedirect(index, slot);

        if (this.isOpen() && this.autoUpdate) {
            this.screenHandler.setSlot(GUI_TO_VANILLA_IDS[index], slot);
        }
        this.hasRedirects = true;
    }

    @Override
    public void clearSlot(int index) {
        super.clearSlot(index);

        if (this.isOpen() && this.autoUpdate) {
            if (this.screenHandler != null) {
                this.screenHandler.setSlot(GUI_TO_VANILLA_IDS[index], new VirtualSlot(this, index, 0, 0));
            }
        }
    }

    @Override
    public boolean click(int index, ClickType type, SlotActionType action) {
        if (index >= 0 && index < SIZE) {
            return super.click(VANILLA_TO_GUI_IDS[index], type, action);
        }
        return super.click(index, type, action);
    }

    @Override
    public int getHotbarSlotIndex(int slots, int index) {
        // We add the offhand before the inventory, so we need to shift by -1
        return super.getHotbarSlotIndex(slots, index) - 1;
    }

    @Override
    public int getOffhandSlotIndex() {
        return 9;
    }

    @Override
    public boolean open() {
        if (this.player.isDisconnected() || this.isOpen()) {
            return false;
        } else {
            this.beforeOpen();
            this.openScreenHandler();
            this.afterOpen();
            return true;
        }
    }

    private void openScreenHandler() {
        this.onOpen();

        if (this.player.currentScreenHandler != this.player.playerScreenHandler && this.player.currentScreenHandler != this.screenHandler) {
            this.player.closeHandledScreen();
        }

        if (this.screenHandler == null) {
            this.screenHandler = new HotbarScreenHandler(null, 0, this, this.player);
        }

        this.player.currentScreenHandler = this.screenHandler;
        ((ServerPlayerEntityAccessor) this.player).callOnScreenHandlerOpened(this.screenHandler);

        this.player.networkHandler.sendPacket(new UpdateSelectedSlotS2CPacket(this.selectedSlot));
    }

    /**
     * It's run after player changes selected slot. It can also block switching by returning false
     *
     * @param slot the newly selected slot
     * @return true to allow or false for canceling switching
     */
    public boolean onSelectedSlotChange(int slot) {
        this.selectedSlot = MathHelper.clamp(slot, 0, 8);
        return true;
    }

    /**
     * This method is called when player uses an item.
     * The vanilla action is always canceled.
     */
    public void onClickItem() {
        if (this.player.isSneaking()) {
            this.tickLimitedClick(this.selectedSlot, ClickType.MOUSE_RIGHT_SHIFT, SlotActionType.QUICK_MOVE);
        } else {
            this.tickLimitedClick(this.selectedSlot, ClickType.MOUSE_RIGHT, SlotActionType.PICKUP);
        }
    }

    /**
     * This method is called when player swings their arm
     * If you return false, vanilla action will be canceled
     */
    public boolean onHandSwing() {
        if (this.player.isSneaking()) {
            this.tickLimitedClick(this.selectedSlot, ClickType.MOUSE_LEFT_SHIFT, SlotActionType.QUICK_MOVE);
        } else {
            this.tickLimitedClick(this.selectedSlot, ClickType.MOUSE_LEFT, SlotActionType.PICKUP);
        }
        return false;
    }

    public boolean onPickItemFromBlock(BlockPos pos, boolean includeData) {
        this.tickLimitedClick(this.selectedSlot, ClickType.MOUSE_MIDDLE, SlotActionType.CLONE);
        return false;
    }

    public boolean onPickItemFromEntity(int entityId, boolean includeData) {
        this.tickLimitedClick(this.selectedSlot, ClickType.MOUSE_MIDDLE, SlotActionType.CLONE);
        return false;
    }

    /**
     * This method is called when player clicks block
     * If you return false, vanilla action will be canceled
     */
    public boolean onClickBlock(BlockHitResult hitResult) {
        if (this.player.isSneaking()) {
            this.tickLimitedClick(this.selectedSlot, ClickType.MOUSE_RIGHT_SHIFT, SlotActionType.QUICK_MOVE);
        } else {
            this.tickLimitedClick(this.selectedSlot, ClickType.MOUSE_RIGHT, SlotActionType.PICKUP);
        }

        return false;
    }

    /**
     * This method is called when player send PlayerAction packet
     * If you return false, vanilla action will be canceled
     */
    public boolean onPlayerAction(PlayerActionC2SPacket.Action action, Direction direction) {
        switch (action) {
            case DROP_ITEM -> this.tickLimitedClick(this.selectedSlot, ClickType.DROP, SlotActionType.THROW);
            case DROP_ALL_ITEMS -> this.tickLimitedClick(this.selectedSlot, ClickType.CTRL_DROP, SlotActionType.THROW);
            case STOP_DESTROY_BLOCK -> {
                if (this.player.isSneaking()) {
                    this.tickLimitedClick(this.selectedSlot, ClickType.MOUSE_LEFT_SHIFT, SlotActionType.QUICK_MOVE);
                } else {
                    this.tickLimitedClick(this.selectedSlot, ClickType.MOUSE_LEFT, SlotActionType.PICKUP);
                }
            }
            case SWAP_ITEM_WITH_OFFHAND -> this.tickLimitedClick(this.selectedSlot, ClickType.OFFHAND_SWAP, SlotActionType.SWAP);
        }

        return false;
    }

    /**
     * This method is called when player clicks an entity
     * If you return false, vanilla action will be canceled
     */
    public boolean onClickEntity(int entityId, EntityInteraction type, boolean isSneaking, @Nullable Vec3d interactionPos) {
        if (type == EntityInteraction.ATTACK) {
            if (isSneaking) {
                this.tickLimitedClick(this.selectedSlot, ClickType.MOUSE_LEFT_SHIFT, SlotActionType.QUICK_MOVE);
            } else {
                this.tickLimitedClick(this.selectedSlot, ClickType.MOUSE_LEFT, SlotActionType.PICKUP);
            }
        } else {
            if (isSneaking) {
                this.tickLimitedClick(this.selectedSlot, ClickType.MOUSE_RIGHT_SHIFT, SlotActionType.QUICK_MOVE);
            } else {
                this.tickLimitedClick(this.selectedSlot, ClickType.MOUSE_RIGHT, SlotActionType.PICKUP);
            }
        }
        return false;
    }

    /**
     * @return selectedSlot
     */
    public int getSelectedSlot() {
        return this.selectedSlot;
    }

    /**
     * Changes selected slot and sends it to player
     *
     * @param value slot between 0 and 8
     */
    public void setSelectedSlot(int value) {
        this.selectedSlot = MathHelper.clamp(value, 0, 8);
        if (this.isOpen()) {
            this.player.networkHandler.sendPacket(new UpdateSelectedSlotS2CPacket(this.selectedSlot));
        }
    }

    @ApiStatus.Internal
    private void tickLimitedClick(int selectedSlot, ClickType type, SlotActionType actionType) {
        if (this.clicksPerTick == 0) {
            this.click(GUI_TO_VANILLA_IDS[selectedSlot], type, actionType);
        }
        this.clicksPerTick++;
    }

    @Override
    public void onTick() {
        this.clicksPerTick = 0;
        super.onTick();
    }

    @Override
    public void close(boolean screenHandlerIsClosed) {
        if ((this.isOpen() || screenHandlerIsClosed) && !this.reOpen) {
            if (!screenHandlerIsClosed && this.player.currentScreenHandler == this.screenHandler) {
                this.player.closeHandledScreen();
                this.player.networkHandler.sendPacket(new UpdateSelectedSlotS2CPacket(this.player.getInventory().getSelectedSlot()));
            }

            this.player.currentScreenHandler.syncState();

            this.onClose();
        } else {
            this.reOpen = false;
        }
    }

    @Override
    public boolean isIncludingPlayer() {
        return true;
    }

    @Override
    public int getVirtualSize() {
        return SIZE;
    }

    @Override
    public boolean isRedirectingSlots() {
        return this.hasRedirects;
    }

    @Override
    public int getSyncId() {
        return 0;
    }

    @Override
    public int getSize() {
        return SIZE;
    }

    @Deprecated
    @Override
    public int getHeight() {
        return 4;
    }

    @Deprecated
    @Override
    public int getWidth() {
        return 9;
    }

    @Deprecated
    @Override
    public ScreenHandlerType<?> getType() {
        return null;
    }

    @Deprecated
    @Override
    public boolean getLockPlayerInventory() {
        return true;
    }

    @Deprecated
    @Override
    public void setLockPlayerInventory(boolean value) {

    }

    @Deprecated
    @Override
    public @Nullable Text getTitle() {
        return null;
    }

    @Deprecated
    @Override
    public void setTitle(Text title) {

    }

    public enum EntityInteraction {
        INTERACT,
        ATTACK,
        INTERACT_AT
    }
}
