package xyz.nucleoid.plasmid.api.game.common.team;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterators;
import it.unimi.dsi.fastutil.objects.Object2ObjectLinkedOpenHashMap;
import it.unimi.dsi.fastutil.objects.Object2ObjectOpenHashMap;
import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import xyz.nucleoid.plasmid.api.game.player.MutablePlayerSet;
import xyz.nucleoid.plasmid.api.game.player.PlayerSet;
import xyz.nucleoid.plasmid.api.game.GameActivity;
import xyz.nucleoid.plasmid.api.game.GameSpace;
import xyz.nucleoid.plasmid.api.game.event.GamePlayerEvents;
import xyz.nucleoid.plasmid.mixin.chat.PlayerListS2CPacketEntryAccessor;
import xyz.nucleoid.plasmid.api.util.PlayerRef;
import xyz.nucleoid.stimuli.event.EventResult;
import xyz.nucleoid.stimuli.event.player.PlayerDamageEvent;

import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import net.minecraft.class_1282;
import net.minecraft.class_2561;
import net.minecraft.class_2583;
import net.minecraft.class_2596;
import net.minecraft.class_268;
import net.minecraft.class_269;
import net.minecraft.class_2703;
import net.minecraft.class_3222;
import net.minecraft.class_5900;

/**
 * Simple, {@link GameActivity} specific team manager class.
 */
@SuppressWarnings({ "unused" })
public final class TeamManager implements Iterable<GameTeam> {
    private final Map<GameTeamKey, State> teamToState = new Object2ObjectLinkedOpenHashMap<>();
    private final Map<UUID, GameTeamKey> playerToTeam = new Object2ObjectOpenHashMap<>();

    private final class_269 scoreboard = new class_269();
    private final GameSpace gameSpace;

    private boolean applyNameFormatting = true;

    private TeamManager(GameSpace gameSpace) {
        this.gameSpace = gameSpace;
    }

    /**
     * Creates and applies a {@link TeamManager} instance to the given {@link GameActivity}.
     *
     * @param activity the activity to apply teams to
     * @return the constructed {@link TeamManager}
     */
    public static TeamManager addTo(GameActivity activity) {
        var manager = new TeamManager(activity.getGameSpace());
        activity.listen(GamePlayerEvents.ADD, manager::onAddPlayer);
        activity.listen(GamePlayerEvents.REMOVE, manager::onRemovePlayer);
        activity.listen(PlayerDamageEvent.EVENT, manager::onDamagePlayer);
        activity.listen(GamePlayerEvents.DISPLAY_NAME, manager::onFormatDisplayName);
        return manager;
    }

    /**
     * Registers a team to this {@link TeamManager}.
     * Note that attempting to use an unregistered team will throw an exception!
     *
     * @param key an identifier for the team to add
     * @param config the configuration for the given team
     * @return {@code true} if team is registered for the first time
     */
    public boolean addTeam(GameTeamKey key, GameTeamConfig config) {
        return this.addTeam(new GameTeam(key, config));
    }

    /**
     * Registers a team to this {@link TeamManager}.
     * Note that attempting to use an unregistered team will throw an exception!
     *
     * @param team the {@link GameTeam} to add
     * @return {@code true} if team is registered for the first time
     */
    public boolean addTeam(GameTeam team) {
        return this.teamToState.putIfAbsent(team.key(), new State(team)) == null;
    }

    /**
     * Registers a collection of teams to this {@link TeamManager}.
     * Note that attempting to use an unregistered team will throw an exception!
     *
     * @param teams the collection of teams to add
     */
    public void addTeams(GameTeamList teams) {
        teams.forEach(this::addTeam);
    }

    /**
     * Updates the {@link GameTeamConfig} associated with the given {@link GameTeamKey}.
     * These changes will then be synced to players and applied immediately.
     *
     * @param team the {@link GameTeamKey} to modify
     * @param config the new {@link GameTeamConfig} to apply to this team
     */
    public void setTeamConfig(GameTeamKey team, GameTeamConfig config) {
        this.teamState(team).setConfig(config);
        this.sendTeamUpdates(team);
    }

    /**
     * Gets the associated {@link GameTeamConfig} for the given {@link GameTeamKey}.
     * Attempting to access a team that is not registered will throw an exception!
     *
     * @param team the team to query
     * @return the associated {@link GameTeamConfig}
     */
    public GameTeamConfig getTeamConfig(GameTeamKey team) {
        return this.teamState(team).team.config();
    }

    /**
     * Adds given player to the given team, and removes them from any previous team they were apart of.
     *
     * @param player {@link PlayerRef} to add
     * @param team the team to add the player to
     * @return {@code true} if player was successfully added
     */
    public boolean addPlayerTo(PlayerRef player, GameTeamKey team) {
        var lastTeam = this.playerToTeam.get(player.id());
        if (lastTeam == team) {
            return false;
        }

        if (lastTeam != null) {
            this.removePlayerFrom(player, lastTeam);
        }

        this.playerToTeam.put(player.id(), team);
        for (var gameSpacePlayer : gameSpace.getPlayers()) {
            this.sendTeamsToPlayer(gameSpacePlayer);
        }

        var state = this.teamState(team);
        if (state.allPlayers.add(player)) {
            var entity = this.gameSpace.getPlayers().getEntity(player.id());
            if (entity != null) {
                this.addOnlinePlayer(entity, state);
            }
            return true;
        } else {
            return false;
        }
    }

    /**
     * Adds given player to the given team, and removes them from any previous team they were apart of.
     *
     * @param player {@link class_3222} to add
     * @param team the team to add the player to
     * @return {@code true} if player was successfully added
     */
    public boolean addPlayerTo(class_3222 player, GameTeamKey team) {
        return this.addPlayerTo(PlayerRef.of(player), team);
    }

    /**
     * Removes the given player from the given team.
     *
     * @param player the {@link class_3222} of the player to remove
     * @param team the team to be removed from
     * @return {@code true} if the player was removed from this team
     */
    public boolean removePlayerFrom(class_3222 player, GameTeamKey team) {
        return this.removePlayerFrom(PlayerRef.of(player), team);
    }

    /**
     * Removes the given player from the given team.
     *
     * @param player the {@link PlayerRef} of the player to remove
     * @param team the team to be removed from
     * @return {@code true} if the player was removed from this team
     */
    public boolean removePlayerFrom(PlayerRef player, GameTeamKey team) {
        if (!this.playerToTeam.remove(player.id(), team)) {
            return false;
        }

        var state = this.teamState(team);
        if (!state.allPlayers.remove(player)) {
            throw new IllegalStateException("Player " + player + " was not in team " + team + ", but had a mapping");
        }

        var entity = state.onlinePlayers.getEntity(player.id());
        if (entity != null) {
            this.sendRemoveTeamsForPlayer(entity);
            this.removeOnlinePlayer(entity, state);
        }
        return true;
    }

    /**
     * Removes the given player from any team they are apart of.
     *
     * @param player the {@link class_3222} of the player to remove
     * @return the team that the player was removed from, or {@code null}
     */
    @Nullable
    public GameTeamKey removePlayer(class_3222 player) {
        return this.removePlayer(PlayerRef.of(player));
    }

    /**
     * Removes the given player from any team they are apart of.
     *
     * @param player the {@link PlayerRef} of the player to remove
     * @return the team that the player was removed from, or {@code null}
     */
    @Nullable
    public GameTeamKey removePlayer(PlayerRef player) {
        var team = this.teamFor(player);
        if (team != null) {
            this.removePlayerFrom(player, team);
        }
        return team;
    }

    /**
     * Returns the team that the given player is apart of.
     *
     * @param player the player to query
     * @return the player's {@link GameTeamKey} or {@code null}
     */
    @Nullable
    public GameTeamKey teamFor(PlayerRef player) {
        return this.playerToTeam.get(player.id());
    }

    /**
     * Returns the team that the given player is apart of.
     *
     * @param player the player to query
     * @return the player's {@link GameTeamKey} or {@code null}
     */
    @Nullable
    public GameTeamKey teamFor(class_3222 player) {
        return this.playerToTeam.get(player.method_5667());
    }

    /**
     * Gets the {@link PlayerSet} of all online players within the given team.
     *
     * @param team targeted {@link GameTeamKey}
     * @return a {@link PlayerSet} of all online players within the given team
     */
    public PlayerSet playersIn(GameTeamKey team) {
        return this.teamState(team).onlinePlayers;
    }

    /**
     * Gets the {@link Set<PlayerRef>} of all players (including offline!) within the given team.
     *
     * @param team targeted {@link GameTeamKey}
     * @return a {@link Set<PlayerRef>} of all players within the given team
     */
    public Set<PlayerRef> allPlayersIn(GameTeamKey team) {
        return this.teamState(team).allPlayers;
    }

    private class_2561 formatPlayerName(class_3222 player, class_2561 name) {
        var team = this.teamFor(player);
        if (team != null) {
            var config = this.teamState(team).team.config();
            var style = class_2583.field_24360.method_27706(config.chatFormatting());
            return class_2561.method_43473().method_10852(config.prefix())
                    .method_10852(name.method_27661().method_10862(style))
                    .method_10852(config.suffix());
        }
        return name;
    }

    @Nullable
    public GameTeamKey getSmallestTeam() {
        GameTeamKey smallest = null;
        int count = Integer.MAX_VALUE;

        for (var state : this.teamToState.values()) {
            int size = state.onlinePlayers.size();
            if (size <= count) {
                smallest = state.team.key();
                count = size;
            }
        }

        return smallest;
    }

    public void enableNameFormatting() {
        this.applyNameFormatting = true;
    }

    public void disableNameFormatting() {
        this.applyNameFormatting = false;
    }

    @NotNull
    private TeamManager.State teamState(GameTeamKey team) {
        return Preconditions.checkNotNull(this.teamToState.get(team), "unregistered team for " + team);
    }

    private void onAddPlayer(class_3222 player) {
        this.sendTeamsToPlayer(player);
        this.restoreFormerTeams(player);
    }

    private void restoreFormerTeams(class_3222 player) {
        var team = this.teamFor(player);
        if (team != null) {
            var state = this.teamState(team);
            this.addOnlinePlayer(player, state);
        }
    }

    private void onRemovePlayer(class_3222 player) {
        var team = this.teamFor(player);
        if (team != null) {
            var state = this.teamState(team);
            this.removeOnlinePlayer(player, state);
        }

        if (!player.method_14239()) {
            this.sendRemoveTeamsForPlayer(player);
        }
    }

    private EventResult onDamagePlayer(class_3222 player, class_1282 source, float amount) {
        if (source.method_5529() instanceof class_3222 attacker) {
            var playerTeam = this.teamFor(player);
            var attackerTeam = this.teamFor(attacker);

            if (playerTeam != null && playerTeam == attackerTeam && !this.getTeamConfig(playerTeam).friendlyFire()) {
                return EventResult.DENY;
            }
        }

        return EventResult.PASS;
    }

    private class_2561 onFormatDisplayName(class_3222 player, class_2561 name, class_2561 vanilla) {
        return this.applyNameFormatting ? this.formatPlayerName(player, name) : name;
    }

    private void sendTeamsToPlayer(class_3222 player) {
        for (var state : this.teamToState.values()) {
            player.field_13987.method_14364(class_5900.method_34172(state.scoreboardTeam, true));
            for (var member : state.onlinePlayers) {
                player.field_13987.method_14364(this.updatePlayerName(member));
            }
        }
    }

    private void sendRemoveTeamsForPlayer(class_3222 player) {
        for (var state : this.teamToState.values()) {
            player.field_13987.method_14364(class_5900.method_34170(state.scoreboardTeam));

            for (var member : state.onlinePlayers) {
                player.field_13987.method_14364(this.resetPlayerName(member));
            }
        }
    }

    private void addOnlinePlayer(class_3222 player, State state) {
        if (!state.allPlayers.contains(PlayerRef.of(player))) {
            throw new IllegalStateException("Tried to mark player " + player.method_5820() + " as online in team " + state.team + ", but they are not in this team");
        }

        state.onlinePlayers.add(player);
        state.scoreboardTeam.method_1204().add(player.method_5820());

        this.sendPacketToAll(this.changePlayerTeam(player, state, class_5900.class_5901.field_29155));
        this.sendPacketToAll(this.resetPlayerName(player));
    }

    private void removeOnlinePlayer(class_3222 player, State state) {
        if (!state.onlinePlayers.remove(player)) {
            throw new IllegalStateException("Tried to mark player " + player.method_5820() + " as offline in team " + state.team + ", but they were not online in this team");
        }
        state.scoreboardTeam.method_1204().remove(player.method_5820());

        this.sendPacketToAll(this.changePlayerTeam(player, state, class_5900.class_5901.field_29156));
        this.sendPacketToAll(this.resetPlayerName(player));
    }

    private void sendTeamUpdates(GameTeamKey gameTeamKey) {
        var state = this.teamState(gameTeamKey);
        this.sendPacketToAll(class_5900.method_34172(state.scoreboardTeam, true));
    }

    private class_5900 changePlayerTeam(class_3222 player, State team, class_5900.class_5901 operation) {
        return class_5900.method_34171(team.scoreboardTeam, player.method_7334().name(), operation);
    }

    private class_2703 updatePlayerName(class_3222 player) {
        var packet = new class_2703(class_2703.class_5893.field_29139, player);

        var entry = packet.method_46329().get(0);
        var name = player.method_14206();
        if (name == null) {
            name = player.method_5477();
        }
        ((PlayerListS2CPacketEntryAccessor) (Object) entry).setDisplayName(this.formatPlayerName(player, name));

        return packet;
    }

    private class_2703 resetPlayerName(class_3222 player) {
        return new class_2703(class_2703.class_5893.field_29139, player);
    }

    private void sendPacketToAll(class_2596<?> packet) {
        this.gameSpace.getPlayers().sendPacket(packet);
    }

    @NotNull
    @Override
    public Iterator<GameTeam> iterator() {
        return Iterators.transform(this.teamToState.values().iterator(), state -> state.team);
    }

    final class State {
        final Set<PlayerRef> allPlayers;
        final MutablePlayerSet onlinePlayers;
        final class_268 scoreboardTeam;

        GameTeam team;

        State(GameTeam team) {
            this.allPlayers = new ObjectOpenHashSet<>();
            this.onlinePlayers = new MutablePlayerSet(TeamManager.this.gameSpace.getServer());

            this.scoreboardTeam = new class_268(TeamManager.this.scoreboard, team.key().id());
            team.config().applyToScoreboard(this.scoreboardTeam);

            this.team = team;
        }

        public void setConfig(GameTeamConfig config) {
            this.team = this.team.withConfig(config);
            config.applyToScoreboard(this.scoreboardTeam);
        }
    }
}


