package xyz.nucleoid.plasmid.api.util;

import it.unimi.dsi.fastutil.longs.LongArrayFIFOQueue;
import it.unimi.dsi.fastutil.longs.LongOpenHashSet;
import it.unimi.dsi.fastutil.longs.LongSet;
import it.unimi.dsi.fastutil.shorts.ShortArrayFIFOQueue;
import java.util.ArrayList;
import java.util.function.Consumer;
import net.minecraft.class_2338;
import net.minecraft.class_2350;
import net.minecraft.class_2382;

/**
 * Provides functionality for traversing the graph of the 3D block grid.
 * <p>
 * This is particularly useful for implementing algorithms such as flood-fill.
 */
public final class BlockTraversal {
    private Connectivity connectivity = Connectivity.SIX;
    private Order order = Order.BREADTH_FIRST;

    private BlockTraversal() {
    }

    /**
     * Creates a {@link BlockTraversal} instance with default settings.
     * This implies six-connectivity and breadth-first traversal order.
     *
     * @return a new {@link BlockTraversal} instance
     */
    public static BlockTraversal create() {
        return new BlockTraversal();
    }

    /**
     * Sets the type of {@link Connectivity} to use for block traversal.
     * Is set to {@link Connectivity#SIX} by default.
     *
     * @return this {@link BlockTraversal} instance for chaining
     * @see <a href="https://en.wikipedia.org/wiki/Pixel_connectivity#3-dimensional">3-dimensional pixel connectivity</a>
     */
    public BlockTraversal connectivity(Connectivity connectivity) {
        this.connectivity = connectivity;
        return this;
    }

    /**
     * Sets the {@link Order} to use for block traversal.
     * Is set to {@link Order#BREADTH_FIRST} by default.
     *
     * @return this {@link BlockTraversal} instance for chaining
     * @see Order
     * @see <a href="https://en.wikipedia.org/wiki/Breadth-first_search">Breadth-first search</a>
     * @see <a href="https://en.wikipedia.org/wiki/Depth-first_search">Depth-first search</a>
     */
    public BlockTraversal order(Order order) {
        this.order = order;
        return this;
    }

    /**
     * Traverses the block graph from the given origin by calling the given {@link Visitor}.
     * <p>
     * The given visitor should control how traversal exits. If it does not return {@link Result#TERMINATE}, blocks
     * will be traversed infinitely!
     *
     * @param origin the origin block position to start traversing from
     * @param visitor the visitor to call with each traversed position
     */
    public void accept(class_2338 origin, Visitor visitor) {
        var state = new State(this.order);
        state.enqueue(origin, origin, 0);

        var pos = new class_2338.class_2339();
        var fromPos = new class_2338.class_2339();

        var offsets = this.connectivity.offsets;

        while (!state.isComplete()) {
            pos.method_16363(state.dequeuePos());
            fromPos.method_16363(state.dequeueFromPos());
            int depth = state.dequeueDepth();

            if (state.tryVisit(pos) && visitor.visit(pos, fromPos, depth) == Result.CONTINUE) {
                int nextDepth = depth + 1;
                fromPos.method_10101(pos);

                for (var offset : offsets) {
                    pos.method_25504(fromPos, offset.method_10263(), offset.method_10264(), offset.method_10260());
                    state.enqueue(pos, fromPos, nextDepth);
                }
            }
        }
    }

    static final class State {
        private final Order order;

        private final LongSet visited = new LongOpenHashSet();
        private final LongArrayFIFOQueue queue = new LongArrayFIFOQueue();
        private final LongArrayFIFOQueue fromQueue = new LongArrayFIFOQueue();
        private final ShortArrayFIFOQueue depthQueue = new ShortArrayFIFOQueue();

        State(Order order) {
            this.order = order;
        }

        boolean tryVisit(class_2338 pos) {
            return this.visited.add(pos.method_10063());
        }

        void enqueue(class_2338 pos, class_2338 from, int depth) {
            this.queue.enqueue(pos.method_10063());
            this.fromQueue.enqueue(from.method_10063());
            this.depthQueue.enqueue((short) depth);
        }

        long dequeuePos() {
            return this.dequeuePos(this.queue);
        }

        long dequeueFromPos() {
            return this.dequeuePos(this.fromQueue);
        }

        long dequeuePos(LongArrayFIFOQueue queue) {
            return this.order == Order.BREADTH_FIRST ? queue.dequeueLong() : queue.dequeueLastLong();
        }

        int dequeueDepth() {
            return this.order == Order.BREADTH_FIRST ? this.depthQueue.dequeueShort() : this.depthQueue.dequeueLastShort();
        }

        boolean isComplete() {
            return this.queue.isEmpty();
        }
    }

    /**
     * @see <a href="https://en.wikipedia.org/wiki/Pixel_connectivity#3-dimensional">3-dimensional pixel connectivity</a>
     */
    public static final class Connectivity {
        public static final Connectivity SIX = create(Connectivity::generateSix);
        public static final Connectivity EIGHTEEN = create(Connectivity::generateEighteen);
        public static final Connectivity TWENTY_SIX = create(Connectivity::generateTwentySix);

        private static final Connectivity FOUR_X = create(consumer -> generateFour(class_2350.class_2351.field_11048, consumer));
        private static final Connectivity FOUR_Y = create(consumer -> generateFour(class_2350.class_2351.field_11052, consumer));
        private static final Connectivity FOUR_Z = create(consumer -> generateFour(class_2350.class_2351.field_11051, consumer));

        private static final Connectivity EIGHT_X = create(consumer -> generateEight(class_2350.class_2351.field_11048, consumer));
        private static final Connectivity EIGHT_Y = create(consumer -> generateEight(class_2350.class_2351.field_11052, consumer));
        private static final Connectivity EIGHT_Z = create(consumer -> generateEight(class_2350.class_2351.field_11051, consumer));

        final class_2382[] offsets;

        Connectivity(class_2382[] offsets) {
            this.offsets = offsets;
        }

        public static Connectivity four(class_2350.class_2351 orthogonalAxis) {
            return switch (orthogonalAxis) {
                case field_11048 -> FOUR_X;
                case field_11052 -> FOUR_Y;
                case field_11051 -> FOUR_Z;
            };
        }

        public static Connectivity eight(class_2350.class_2351 orthogonalAxis) {
            return switch (orthogonalAxis) {
                case field_11048 -> EIGHT_X;
                case field_11052 -> EIGHT_Y;
                case field_11051 -> EIGHT_Z;
            };
        }

        static Connectivity create(Consumer<Consumer<class_2382>> generator) {
            var offsets = new ArrayList<class_2382>();
            generator.accept(offsets::add);
            return new Connectivity(offsets.toArray(new class_2382[0]));
        }

        private static void generateSix(Consumer<class_2382> consumer) {
            for (var direction : class_2350.values()) {
                consumer.accept(direction.method_62675());
            }
        }

        private static void generateEighteen(Consumer<class_2382> consumer) {
            generateSix(consumer);

            for (int x = -1; x <= 1; x += 2) {
                for (int y = -1; y <= 1; y += 2) {
                    consumer.accept(new class_2338(x, y, 0));
                    consumer.accept(new class_2338(0, x, y));
                    consumer.accept(new class_2338(y, 0, x));
                }
            }
        }

        private static void generateTwentySix(Consumer<class_2382> consumer) {
            generateEighteen(consumer);

            for (int z = -1; z <= 1; z += 2) {
                for (int x = -1; x <= 1; x += 2) {
                    for (int y = -1; y <= 1; y += 2) {
                        consumer.accept(new class_2338(x, y, z));
                    }
                }
            }
        }

        private static void generateFour(class_2350.class_2351 orthogonalAxis, Consumer<class_2382> consumer) {
            for (var direction : class_2350.values()) {
                if (direction.method_10166() != orthogonalAxis) {
                    consumer.accept(direction.method_62675());
                }
            }
        }

        private static void generateEight(class_2350.class_2351 orthogonalAxis, Consumer<class_2382> consumer) {
            generateFour(orthogonalAxis, consumer);

            for (int x = -1; x <= 1; x += 2) {
                for (int y = -1; y <= 1; y += 2) {
                    consumer.accept(switch (orthogonalAxis) {
                        case field_11048 -> new class_2338(0, x, y);
                        case field_11052 -> new class_2338(y, 0, x);
                        case field_11051 -> new class_2338(x, y, 0);
                    });
                }
            }
        }
    }

    /**
     * @see <a href="https://en.wikipedia.org/wiki/Graph_traversal">Graph Traversal</a>
     */
    public enum Order {
        /**
         * Breadth-first search traverses all neighbors at the current depth before proceeding to the next depth level.
         *
         * @see <a href="https://en.wikipedia.org/wiki/Breadth-first_search">Breadth-first search</a>
         */
        BREADTH_FIRST,
        /**
         * Depth-first search traverses each branch as far as possible before trying other branches.
         *
         * @see <a href="https://en.wikipedia.org/wiki/Depth-first_search">Depth-first search</a>
         */
        DEPTH_FIRST
    }

    /**
     * Accepts each block traversed by a {@link BlockTraversal} instance and controls how the algorithm should advance.
     */
    public interface Visitor {
        /**
         * Called for each traversed block and determines whether traversal should continue from each block.
         *
         * @param pos the current block position.
         * @param fromPos the block position the current was found from. Can be equal to the current pos for the origin.
         * @param depth the depth at the current block (how many steps we have taken from the origin)
         * @return whether or not traversal should continue from this point.
         */
        Result visit(class_2338 pos, class_2338 fromPos, int depth);
    }

    public enum Result {
        /**
         * Continue traversing from this block and consider all connected edges.
         */
        CONTINUE,
        /**
         * Stop traversing from this block and ignore any connected edges.
         * <p>
         * This will not stop traversal altogether, it just signals that this branch should terminate.
         */
        TERMINATE
    }
}
