package xyz.nucleoid.plasmid.impl.command;

import com.mojang.brigadier.Command;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.context.CommandContext;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import com.mojang.brigadier.exceptions.DynamicCommandExceptionType;
import com.mojang.brigadier.exceptions.SimpleCommandExceptionType;
import com.mojang.logging.LogUtils;
import me.lucko.fabric.api.permissions.v0.Permissions;
import net.minecraft.class_124;
import net.minecraft.class_2168;
import net.minecraft.class_2179;
import net.minecraft.class_2186;
import net.minecraft.class_2509;
import net.minecraft.class_2561;
import net.minecraft.class_3222;
import net.minecraft.class_5250;
import net.minecraft.class_6880;
import org.slf4j.Logger;
import xyz.nucleoid.plasmid.api.registry.PlasmidRegistryKeys;
import xyz.nucleoid.plasmid.impl.Plasmid;
import xyz.nucleoid.plasmid.impl.command.argument.GameConfigArgument;
import xyz.nucleoid.plasmid.impl.command.argument.GameSpaceArgument;
import xyz.nucleoid.plasmid.impl.command.ui.GameJoinUi;
import xyz.nucleoid.plasmid.api.game.GameCloseReason;
import xyz.nucleoid.plasmid.api.game.GameOpenException;
import xyz.nucleoid.plasmid.api.game.GameSpace;
import xyz.nucleoid.plasmid.api.game.GameTexts;
import xyz.nucleoid.plasmid.api.game.config.GameConfig;
import xyz.nucleoid.plasmid.impl.game.manager.GameSpaceManagerImpl;
import xyz.nucleoid.plasmid.api.game.player.GamePlayerJoiner;
import xyz.nucleoid.plasmid.api.game.player.JoinIntent;
import xyz.nucleoid.plasmid.api.util.Scheduler;

import java.util.Comparator;
import java.util.stream.Collectors;

import static net.minecraft.class_2170.method_9244;
import static net.minecraft.class_2170.method_9247;

public final class GameCommand {
    private static final Logger LOGGER = LogUtils.getLogger();

    public static final SimpleCommandExceptionType NO_GAME_OPEN = new SimpleCommandExceptionType(
            class_2561.method_43471("text.plasmid.game.join.no_game_open")
    );

    public static final SimpleCommandExceptionType NOT_IN_GAME = new SimpleCommandExceptionType(
            class_2561.method_43471("text.plasmid.game.not_in_game")
    );

    public static final DynamicCommandExceptionType MALFORMED_CONFIG = new DynamicCommandExceptionType(error ->
            class_2561.method_54159("text.plasmid.game.open.malformed_config", error)
    );

    public static final DynamicCommandExceptionType PLAYER_NOT_IN_GAME = new DynamicCommandExceptionType(player ->
            class_2561.method_54159("text.plasmid.game.locate.player_not_in_game", player)
    );

    // @formatter:off
    public static void register(CommandDispatcher<class_2168> dispatcher) {
        dispatcher.register(
            method_9247("game")
                .then(method_9247("open")
                    .requires(Permissions.require("plasmid.command.game.open", 2))
                    .then(GameConfigArgument.argument("game_config")
                        .executes(GameCommand::openGame)
                    )
                    .then(method_9244("game_config_nbt", class_2179.method_9284())
                        .executes(GameCommand::openAnonymousGame)
                    )
                )
                .then(method_9247("propose")
                    .requires(Permissions.require("plasmid.command.game.propose", 2))
                    .then(GameSpaceArgument.argument("game_space")
                        .executes(GameCommand::proposeGame)
                    )
                        .executes(GameCommand::proposeCurrentGame)
                )
                .then(method_9247("start")
                    .requires(Permissions.require("plasmid.command.game.start", 2))
                    .executes(GameCommand::startGame)
                )
                .then(method_9247("stop")
                    .requires(Permissions.require("plasmid.command.game.stop", 2))
                    .executes(GameCommand::stopGame)
                        .then(method_9247("confirm")
                            .executes(GameCommand::stopGameConfirmed)
                        )
                )
                .then(method_9247("kick")
                    .requires(Permissions.require("plasmid.command.game.kick", 2))
                    .then(method_9244("targets", class_2186.method_9308())
                        .executes(GameCommand::kickPlayers)
                    )
                )
                .then(method_9247("join")
                    .executes(ctx -> GameCommand.joinGame(ctx, JoinIntent.PLAY))
                    .then(GameSpaceArgument.argument("game_space")
                        .executes(ctx -> GameCommand.joinQualifiedGame(ctx, JoinIntent.PLAY))
                    )
                )
                .then(method_9247("spectate")
                     .executes(ctx -> GameCommand.joinGame(ctx, JoinIntent.SPECTATE))
                     .then(GameSpaceArgument.argument("game_space")
                          .executes(ctx -> GameCommand.joinQualifiedGame(ctx, JoinIntent.SPECTATE))
                     )
                )
                .then(method_9247("joinall")
                    .requires(Permissions.require("plasmid.command.game.joinall", 2))
                    .executes(GameCommand::joinAllGame)
                    .then(GameSpaceArgument.argument("game_space")
                        .executes(GameCommand::joinAllQualifiedGame)
                    )
                )
                .then(method_9247("locate")
                        .then(method_9244("player", class_2186.method_9305())
                        .executes(GameCommand::locatePlayer))
                )
                .then(method_9247("leave").executes(GameCommand::leaveGame))
                .then(method_9247("list").executes(GameCommand::listGames))
        );
    }
    // @formatter:on

    private static int openGame(CommandContext<class_2168> context) throws CommandSyntaxException {
        return openGame(context, false);
    }

    protected static int openGame(CommandContext<class_2168> context, boolean test) throws CommandSyntaxException {
        try {
            var game = GameConfigArgument.get(context, "game_config");
            return openGame(context, game, test);
        } catch (CommandSyntaxException e) {
            throw e;
        } catch (Exception e) {
            LOGGER.error("An unexpected error occurred while opening a game", e);
            context.getSource().method_9226(() -> class_2561.method_43471("text.plasmid.game.open.error").method_27692(class_124.field_1061), false);
            return 0;
        }
    }

    private static int openAnonymousGame(CommandContext<class_2168> context) throws CommandSyntaxException {
        return openAnonymousGame(context, false);
    }

    protected static int openAnonymousGame(CommandContext<class_2168> context, boolean test) throws CommandSyntaxException {
        try {
            var configNbt = class_2179.method_9285(context, "game_config_nbt");
            var game = GameConfig.DIRECT_CODEC.parse(context.getSource().method_30497().method_57093(class_2509.field_11560), configNbt)
                    .getOrThrow(MALFORMED_CONFIG::create);
            return openGame(context, class_6880.method_40223(game), test);
        } catch (CommandSyntaxException e) {
            throw e;
        } catch (Exception e) {
            LOGGER.error("An unexpected error occurred while opening a game", e);
            context.getSource().method_9226(() -> class_2561.method_43471("text.plasmid.game.open.error").method_27692(class_124.field_1061), false);
            return 0;
        }
    }

    private static int openGame(CommandContext<class_2168> context, class_6880<GameConfig<?>> config, boolean test) {
        var source = context.getSource();
        var server = source.method_9211();
        var player = source.method_44023();

        if (player != null) {
            var currentGameSpace = GameSpaceManagerImpl.get().byPlayer(player);
            if (currentGameSpace != null) {
                if (test) {
                    currentGameSpace.close(GameCloseReason.CANCELED);
                } else {
                    currentGameSpace.getPlayers().kick(player);
                }
            }
        }

        GameSpaceManagerImpl.get().open(config).handleAsync((gameSpace, throwable) -> {
            if (throwable == null) {
                onOpenSuccess(source, gameSpace, player, test);
            } else {
                onOpenError(source, throwable);
            }
            return null;
        }, server);

        return Command.SINGLE_SUCCESS;
    }

    private static void onOpenSuccess(class_2168 source, GameSpace gameSpace, class_3222 player, boolean test) {
        var players = source.method_9211().method_3760();

        var message = test ? GameTexts.Broadcast.gameOpenedTesting(source, gameSpace) : GameTexts.Broadcast.gameOpened(source, gameSpace);
        players.method_43514(message, false);

        if (test) {
            joinAllPlayersToGame(source, gameSpace);

            var startResult = gameSpace.requestStart();

            if (!startResult.isOk()) {
                var error = startResult.errorCopy().method_27692(class_124.field_1061);
                gameSpace.getPlayers().sendMessage(error);
            }
        } else if (player != null) {
            tryJoinGame(player, gameSpace, JoinIntent.PLAY);
        }
    }

    private static void onOpenError(class_2168 source, Throwable throwable) {
        Plasmid.LOGGER.error("Failed to start game", throwable);

        var gameOpenException = GameOpenException.unwrap(throwable);

        class_5250 message;
        if (gameOpenException != null) {
            message = gameOpenException.getReason().method_27661();
        } else {
            message = GameTexts.Broadcast.gameOpenError();
        }

        var players = source.method_9211().method_3760();
        players.method_43514(message.method_27692(class_124.field_1061), false);
    }

    private static int proposeGame(CommandContext<class_2168> context) throws CommandSyntaxException {
        var gameSpace = GameSpaceArgument.get(context, "game_space");
        return proposeGame(context.getSource(), gameSpace);
    }

    private static int proposeCurrentGame(CommandContext<class_2168> context) throws CommandSyntaxException {
        var source = context.getSource();

        var gameSpace = GameSpaceManagerImpl.get().byPlayer(source.method_9207());
        if (gameSpace == null) {
            throw NOT_IN_GAME.create();
        }

        return proposeGame(source, gameSpace);
    }

    private static int proposeGame(class_2168 source, GameSpace gameSpace) {
        var message = GameTexts.Broadcast.propose(source, gameSpace);

        var playerManager = source.method_9211().method_3760();
        playerManager.method_43514(message, false);

        return Command.SINGLE_SUCCESS;
    }

    private static int joinGame(CommandContext<class_2168> context, JoinIntent intent) throws CommandSyntaxException {
        new GameJoinUi(context.getSource().method_9207(), intent).open();
        return Command.SINGLE_SUCCESS;
    }

    private static int joinQualifiedGame(CommandContext<class_2168> context, JoinIntent intent) throws CommandSyntaxException {
        var gameSpace = GameSpaceArgument.get(context, "game_space");
        tryJoinGame(context.getSource().method_9207(), gameSpace, intent);

        return Command.SINGLE_SUCCESS;
    }

    private static int joinAllGame(CommandContext<class_2168> context) throws CommandSyntaxException {
        GameSpace gameSpace = null;

        var player = context.getSource().method_44023();
        if (player != null) {
            gameSpace = GameSpaceManagerImpl.get().byPlayer(player);
        }

        if (gameSpace == null) {
            gameSpace = getJoinableGameSpace();
        }

        joinAllPlayersToGame(context.getSource(), gameSpace);

        return Command.SINGLE_SUCCESS;
    }

    private static int joinAllQualifiedGame(CommandContext<class_2168> context) throws CommandSyntaxException {
        var gameSpace = GameSpaceArgument.get(context, "game_space");
        joinAllPlayersToGame(context.getSource(), gameSpace);

        return Command.SINGLE_SUCCESS;
    }

    private static void joinAllPlayersToGame(class_2168 source, GameSpace gameSpace) {
        var playerManager = source.method_9211().method_3760();

        var players = playerManager.method_14571().stream()
                .filter(player -> !GameSpaceManagerImpl.get().inGame(player))
                .collect(Collectors.toList());

        var intent = JoinIntent.PLAY;
        var result = gameSpace.getPlayers().offer(players, intent);
        if (result.isError()) {
            source.method_9213(result.errorCopy().method_27692(class_124.field_1061));
        }
    }

    private static void tryJoinGame(class_3222 player, GameSpace gameSpace, JoinIntent intent) {
        var result = GamePlayerJoiner.tryJoin(player, gameSpace, intent);
        if (result.isError()) {
            player.method_64398(result.errorCopy().method_27692(class_124.field_1061));
        }
    }

    private static GameSpace getJoinableGameSpace() throws CommandSyntaxException {
        return GameSpaceManagerImpl.get().getOpenGameSpaces().stream()
                .max(Comparator.comparingInt(space -> space.getPlayers().size()))
                .orElseThrow(NO_GAME_OPEN::create);
    }

    private static int locatePlayer(CommandContext<class_2168> context) throws CommandSyntaxException {
        var player = class_2186.method_9315(context, "player");

        var gameSpace = GameSpaceManagerImpl.get().byPlayer(player);
        if (gameSpace == null) {
            throw PLAYER_NOT_IN_GAME.create(player.method_5477());
        }

        context.getSource().method_9226(() -> GameTexts.Command.located(player, gameSpace), false);

        return Command.SINGLE_SUCCESS;
    }

    private static int leaveGame(CommandContext<class_2168> context) throws CommandSyntaxException {
        var source = context.getSource();
        var player = source.method_9207();

        var gameSpace = GameSpaceManagerImpl.get().byPlayer(player);
        if (gameSpace == null) {
            throw NOT_IN_GAME.create();
        }

        Scheduler.INSTANCE.submit(server -> {
            gameSpace.getPlayers().kick(player);
        });

        return Command.SINGLE_SUCCESS;
    }

    private static int startGame(CommandContext<class_2168> context) throws CommandSyntaxException {
        var source = context.getSource();

        var gameSpace = GameSpaceManagerImpl.get().byPlayer(source.method_9207());
        if (gameSpace == null) {
            throw NOT_IN_GAME.create();
        }

        var startResult = gameSpace.requestStart();

        class_2561 message;
        if (startResult.isOk()) {
            message = GameTexts.Start.startedBy(source).method_27692(class_124.field_1080);
        } else {
            message = startResult.errorCopy().method_27692(class_124.field_1061);
        }

        gameSpace.getPlayers().sendMessage(message);

        return Command.SINGLE_SUCCESS;
    }

    private static int stopGame(CommandContext<class_2168> context) throws CommandSyntaxException {
        var source = context.getSource();
        var gameSpace = GameSpaceManagerImpl.get().byPlayer(source.method_9207());
        if (gameSpace == null) {
            throw NOT_IN_GAME.create();
        }

        var playerSet = gameSpace.getPlayers();

        if (playerSet.size() <= 1) {
            stopGameConfirmed(context);
        } else {
            source.method_9226(
                    () -> GameTexts.Stop.confirmStop().method_27692(class_124.field_1065),
                    false
            );
        }

        return Command.SINGLE_SUCCESS;
    }

    private static int stopGameConfirmed(CommandContext<class_2168> context) throws CommandSyntaxException {
        var source = context.getSource();
        var gameSpace = GameSpaceManagerImpl.get().byPlayer(source.method_9207());
        if (gameSpace == null) {
            throw NOT_IN_GAME.create();
        }

        var playerSet = gameSpace.getPlayers().copy(source.method_9211());

        try {
            gameSpace.close(GameCloseReason.CANCELED);

            var message = GameTexts.Stop.stoppedBy(source);
            playerSet.sendMessage(message.method_27692(class_124.field_1080));
        } catch (Throwable throwable) {
            Plasmid.LOGGER.error("Failed to stop game", throwable);

            playerSet.sendMessage(GameTexts.Stop.genericError().method_27692(class_124.field_1061));
        }

        return Command.SINGLE_SUCCESS;
    }

    private static int listGames(CommandContext<class_2168> context) {
        var registry = context.getSource().method_30497().method_30530(PlasmidRegistryKeys.GAME_CONFIG);
        var source = context.getSource();
        source.method_9226(() -> GameTexts.Command.gameList().method_27692(class_124.field_1067), false);

        registry.method_42017().forEach(game -> {
            var id = game.method_40237().method_29177();
            source.method_9226(() -> {
                String command = "/game open " + id;

                var link = GameConfig.name(game).method_27661()
                        .method_10862(GameTexts.commandLinkStyle(command));

                return GameTexts.Command.listEntry(link);
            }, false);
        });

        return Command.SINGLE_SUCCESS;
    }

    private static int kickPlayers(CommandContext<class_2168> context) throws CommandSyntaxException {
        var source = context.getSource();
        var playerManager = source.method_9211().method_3760();

        var targets = class_2186.method_9312(context, "targets");

        int successes = 0;

        for (var target : targets) {
            var gameSpace = GameSpaceManagerImpl.get().byPlayer(target);
            if (gameSpace != null) {
                var message = GameTexts.Kick.kick(source, target).method_27692(class_124.field_1080);
                playerManager.method_43514(message, false);

                Scheduler.INSTANCE.submit(server -> {
                    gameSpace.getPlayers().kick(target);
                });

                successes += 1;
            }
        }

        return successes;
    }
}
